/*
 *  $Id: field.c 29019 2025-12-16 19:35:27Z yeti-dn $
 *  Copyright (C) 2025 David Nečas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "tests/testlibgwy.h"

/* FIXME: The tolerances for various statistical and other functions were just estimated based on what they do, how
 * many operations they may involve and how well conditioned they may be. With GCC, we some fail occasionally (for
 * unlucky seeds). So they are probably quite tight. With CLANG and fast math, we some fail all the time. */

enum {
    SOME_NUMBER = 7121,
};

typedef gdouble (*MaskedStatFunc)(GwyField *field, GwyNield *nield, GwyMaskingType masking,
                                  gint col, gint row, gint width, gint height);
typedef gdouble (*ConstantDataFunc)(gdouble c, gsize n);

static void
assert_field_content(GwyField *field, const gdouble *ref_data, guint ref_n, gdouble eps)
{
    g_assert_true(GWY_IS_FIELD(field));
    guint xres = gwy_field_get_xres(field);
    guint yres = gwy_field_get_yres(field);
    g_assert_cmpuint(xres*yres, ==, ref_n);
    const gdouble *data = gwy_field_get_data_const(field);
    g_assert_nonnull(data);

    if (eps > 0.0) {
#if 0
        for (guint i = 0; i < yres; i++) {
            for (guint j = 0; j < xres; j++) {
                g_printerr("[%g %g]", gwy_field_get_val(field, j, i), ref_data[i*xres + j]);
            }
            g_printerr("\n");
        }
#endif
        for (guint i = 0; i < ref_n; i++)
            g_assert_cmpfloat_with_epsilon(data[i], ref_data[i], eps);
    }
    else {
        for (guint i = 0; i < ref_n; i++)
            g_assert_cmpfloat(data[i], ==, ref_data[i]);
    }
}

static void
assert_field_properties(GwyField *field, GwyField *reference)
{
    g_assert_cmpint(gwy_field_get_xres(field), ==, gwy_field_get_xres(reference));
    g_assert_cmpint(gwy_field_get_yres(field), ==, gwy_field_get_yres(reference));
    g_assert_cmpfloat(gwy_field_get_xreal(field), ==, gwy_field_get_xreal(reference));
    g_assert_cmpfloat(gwy_field_get_yreal(field), ==, gwy_field_get_yreal(reference));
    g_assert_cmpfloat(gwy_field_get_xoffset(field), ==, gwy_field_get_xoffset(reference));
    g_assert_cmpfloat(gwy_field_get_yoffset(field), ==, gwy_field_get_yoffset(reference));
    g_assert_true(gwy_unit_equal(gwy_field_get_unit_xy(field), gwy_field_get_unit_xy(reference)));
    g_assert_true(gwy_unit_equal(gwy_field_get_unit_z(field), gwy_field_get_unit_z(reference)));
}

void
field_assert_equal(GObject *object, GObject *reference)
{
    g_assert_true(GWY_IS_FIELD(object));
    g_assert_true(GWY_IS_FIELD(reference));

    GwyField *field = GWY_FIELD(object), *field_ref = GWY_FIELD(reference);
    assert_field_properties(field, field_ref);
    assert_field_content(field, gwy_field_get_data(field_ref),
                         gwy_field_get_xres(field_ref)*gwy_field_get_yres(field_ref),
                         0.0);
}

void
test_field_basic(void)
{
    const gdouble zeros[6] = { 0, 0, 0, 0, 0, 0 };
    const gdouble twos[6] = { 2, 2, 2, 2, 2, 2 };

    GwyField *field = gwy_field_new(2, 3, 1.6, 7.5, TRUE);
    assert_field_content(field, zeros, 6, 0.0);
    g_assert_cmpfloat(gwy_field_get_xreal(field), ==, 1.6);
    g_assert_cmpfloat(gwy_field_get_yreal(field), ==, 7.5);
    g_assert_cmpfloat(gwy_field_get_dx(field), ==, 1.6/2);
    g_assert_cmpfloat(gwy_field_get_dy(field), ==, 7.5/3);
    g_assert_cmpfloat(gwy_field_get_xoffset(field), ==, 0);
    g_assert_cmpfloat(gwy_field_get_yoffset(field), ==, 0);

    gwy_field_set_xreal(field, 31);
    gwy_field_set_yreal(field, 39);
    gwy_field_set_xoffset(field, -11);
    gwy_field_set_yoffset(field, 77);
    g_assert_cmpfloat(gwy_field_get_xreal(field), ==, 31.0);
    g_assert_cmpfloat(gwy_field_get_yreal(field), ==, 39.0);
    g_assert_cmpfloat(gwy_field_get_dx(field), ==, 31.0/2);
    g_assert_cmpfloat(gwy_field_get_dy(field), ==, 39.0/3);
    g_assert_cmpfloat(gwy_field_get_xoffset(field), ==, -11);
    g_assert_cmpfloat(gwy_field_get_yoffset(field), ==, 77);

    gwy_field_fill(field, 2.0);
    assert_field_content(field, twos, 6, 0.0);

    gwy_field_clear(field);
    assert_field_content(field, zeros, 6, 0.0);

    g_assert_cmpfloat(gwy_field_get_xreal(field), ==, 31);
    g_assert_cmpfloat(gwy_field_get_yreal(field), ==, 39);
    g_assert_cmpfloat(gwy_field_get_dx(field), ==, 31.0/2);
    g_assert_cmpfloat(gwy_field_get_dy(field), ==, 39.0/3);
    g_assert_cmpfloat(gwy_field_get_xoffset(field), ==, -11);
    g_assert_cmpfloat(gwy_field_get_yoffset(field), ==, 77);

    g_assert_finalize_object(field);
}

void
test_field_invalidate(void)
{
    guint xres = 4, yres = 5, n = xres*yres;
    GwyField *field = gwy_field_new(xres, yres, 1.6, 7.5, TRUE);

    gdouble *data = gwy_field_get_data(field);
    for (guint i = 0; i < n; i++)
        data[i] = i;

    /* This must recompute the mean (which was zero). */
    g_assert_cmpfloat_with_epsilon(gwy_field_get_avg(field), 0.5*(n-1), 0.5*n*DBL_EPSILON);

    for (guint i = 0; i < n; i++)
        data[i] = -3.0*i;

    /* We cannot assert that the mean is *not* recalculated. Recalculating it always is a valid behaviour! */
    gwy_field_invalidate(field);
    g_assert_cmpfloat_with_epsilon(gwy_field_get_avg(field), -1.5*(n-1), 0.5*n*DBL_EPSILON);

    g_assert_finalize_object(field);
}

void
test_field_data_changed(void)
{
    GwyField *field = gwy_field_new(1, 1, 1.0, 1.0, TRUE);
    guint item_changed = 0;
    g_signal_connect_swapped(field, "data-changed", G_CALLBACK(record_signal), &item_changed);
    gwy_field_data_changed(field);
    g_assert_cmpuint(item_changed, ==, 1);
    gwy_field_data_changed(field);
    g_assert_cmpuint(item_changed, ==, 2);
    g_assert_finalize_object(field);
}

static GwyField*
create_field_for_serialisation(void)
{
    GwyField *field = gwy_field_new(13, 7, 26.0, 1.6, FALSE);
    gwy_field_set_xoffset(field, G_LN2);
    gwy_field_set_yoffset(field, GWY_SQRT3);
    gdouble *data = gwy_field_get_data(field);
    for (guint i = 0; i < 13*7; i++)
        data[i] = sqrt(i) - 2*G_PI;

    return field;
}

static void
set_field_units_for_serialisation(GwyField *field)
{
    gwy_unit_set_from_string(gwy_field_get_unit_xy(field), "m");
    gwy_unit_set_from_string(gwy_field_get_unit_z(field), "V");
}

void
test_field_serialization(void)
{
    GwyField *field = create_field_for_serialisation();
    serialize_object_and_back(G_OBJECT(field), field_assert_equal, FALSE, NULL);

    set_field_units_for_serialisation(field);
    serialize_object_and_back(G_OBJECT(field), field_assert_equal, FALSE, NULL);

    g_assert_finalize_object(field);
}

void
test_field_copy(void)
{
    GwyField *field = create_field_for_serialisation();
    serializable_test_copy(GWY_SERIALIZABLE(field), field_assert_equal);

    set_field_units_for_serialisation(field);
    serializable_test_copy(GWY_SERIALIZABLE(field), field_assert_equal);

    g_assert_finalize_object(field);
}

void
test_field_assign(void)
{
    GwyField *field = create_field_for_serialisation();
    serializable_test_assign(GWY_SERIALIZABLE(field), NULL, field_assert_equal);

    set_field_units_for_serialisation(field);
    serializable_test_assign(GWY_SERIALIZABLE(field), NULL, field_assert_equal);

    GwyField *another = gwy_field_new(4, 5, 1.3, 8.0, TRUE);
    gwy_field_set_xoffset(another, -1.0);
    gwy_field_set_yoffset(another, 10.0);
    gwy_unit_set_from_string(gwy_field_get_unit_xy(another), "s");
    gwy_unit_set_from_string(gwy_field_get_unit_z(another), "Hz");
    serializable_test_assign(GWY_SERIALIZABLE(field), GWY_SERIALIZABLE(another), field_assert_equal);
    g_assert_finalize_object(another);

    g_assert_finalize_object(field);
}

void
test_field_resize(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    GwyField *field = gwy_field_new(1, 1, 1.0, 1.0, FALSE);
    for (guint k = 0; k < n; k++) {
        gint width = g_test_rand_int_range(1, 12), height = g_test_rand_int_range(1, 12);
        gwy_field_resize(field, width, height);
        g_assert_cmpint(gwy_field_get_xres(field), ==, width);
        g_assert_cmpint(gwy_field_get_yres(field), ==, height);
        /* Try to crash if the memory is not allocated correctly. */
        gdouble *d = gwy_field_get_data(field);
        gwy_clear(d, width*height);
    }
    g_assert_finalize_object(field);
}

/* NB: Stepping stone.
 *
 * Assuming gwy_field_plane_level() is correct, we then use it to make all kinds of planar fields with known
 * properties. */
void
test_field_plane_level(void)
{
    enum { xres1 = 3, yres1 = 4 };
    GwyField *field1 = gwy_field_new(xres1, yres1, 1.0, 1.0, TRUE);
    gwy_field_plane_level(field1, 2.0, -G_PI, -G_LN2);
    const gdouble expected1[xres1*yres1] = {
        -2.0, -2.0 + G_PI, -2.0 + 2*G_PI,
        -2.0 + G_LN2, -2.0 + G_PI + G_LN2, -2.0 + 2*G_PI + G_LN2,
        -2.0 + 2*G_LN2, -2.0 + G_PI + 2*G_LN2, -2.0 + 2*G_PI + 2*G_LN2,
        -2.0 + 3*G_LN2, -2.0 + G_PI + 3*G_LN2, -2.0 + 2*G_PI + 3*G_LN2,
    };
    /* These should probably be exact. But a few ε is OK. */
    assert_field_content(field1, expected1, xres1*yres1, 4*DBL_EPSILON);
    g_assert_finalize_object(field1);

    enum { xres2 = 2, yres2 = 5 };
    GwyField *field2 = gwy_field_new(xres2, yres2, 2.0, 5.0, TRUE);
    gwy_field_plane_level(field2, -1.0, 1.25, 6.4);
    const gdouble expected2[xres2*yres2] = {
        1.0, 1.0 - 1.25,
        1.0 - 6.4, 1.0 - 1.25 - 6.4,
        1.0 - 2*6.4, 1.0 - 1.25 - 2*6.4,
        1.0 - 3*6.4, 1.0 - 1.25 - 3*6.4,
        1.0 - 4*6.4, 1.0 - 1.25 - 4*6.4,
    };
    /* These should probably be exact. But a few ε is OK. */
    assert_field_content(field2, expected2, xres2*yres2, 20*DBL_EPSILON);
    g_assert_finalize_object(field2);

    enum { xres3 = 6, yres3 = 1 };
    GwyField *field3 = gwy_field_new(xres3, yres3, 0.5, 2.0, TRUE);
    gwy_field_plane_level(field3, 0.5, -2.0, 1e14);
    const gdouble expected3[xres3*yres3] = {
        -0.5, -0.5 + 2.0, -0.5 + 2*2.0, -0.5 + 3*2.0, -0.5 + 4*2.0, -0.5 + 5*2.0,
    };
    /* These should probably be exact. But a few ε is OK. */
    assert_field_content(field3, expected3, xres3*yres3, 8*DBL_EPSILON);
    g_assert_finalize_object(field3);

    enum { xres4 = 1, yres4 = 5 };
    GwyField *field4 = gwy_field_new(xres4, yres4, 1.0, 4.0, TRUE);
    gwy_field_plane_level(field4, -0.5, -1e14, 2.0);
    const gdouble expected4[xres4*yres4] = {
        0.5,
        0.5 - 2.0,
        0.5 - 2*2.0,
        0.5 - 3*2.0,
        0.5 - 4*2.0,
    };
    /* These should probably be exact. But a few ε is OK. */
    assert_field_content(field4, expected4, xres4*yres4, 6*DBL_EPSILON);
    g_assert_finalize_object(field4);
}

static GwyField*
make_random_planar_field(gdouble *pa, gdouble *pbx, gdouble *pby)
{
    gint xres = g_test_rand_int_range(2, 20);
    gint yres = g_test_rand_int_range(2, 20);
    gdouble dx = exp(2*g_test_rand_double() - 1);
    gdouble dy = exp(2*g_test_rand_double() - 1);
    GwyField *field = gwy_field_new(xres, yres, xres*dx, yres*dy, TRUE);
    gdouble a = g_test_rand_double_range(-5.0, 5.0);
    gdouble bx = g_test_rand_double_range(-1.0, 1.0);
    gdouble by = g_test_rand_double_range(-1.0, 1.0);
    gwy_field_plane_level(field, -a, -bx, -by);

    if (pa)
        *pa = a;
    if (pbx)
        *pbx = bx;
    if (pby)
        *pby = by;

    return field;
}

static GwyField*
make_random_bilinear_field(gdouble *pa, gdouble *pbx, gdouble *pby, gdouble *pcxy)
{
    gint xres = g_test_rand_int_range(2, 20);
    gint yres = g_test_rand_int_range(2, 20);
    gdouble dx = exp(2*g_test_rand_double() - 1);
    gdouble dy = exp(2*g_test_rand_double() - 1);
    GwyField *field = gwy_field_new(xres, yres, xres*dx, yres*dy, TRUE);
    gdouble a = g_test_rand_double_range(-5.0, 5.0);
    gdouble bx = g_test_rand_double_range(-1.0, 1.0);
    gdouble by = g_test_rand_double_range(-1.0, 1.0);
    gdouble cxy = g_test_rand_double_range(-0.1, 0.1);

    gdouble *d = gwy_field_get_data(field);
    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            d[i*xres + j] = a + bx*j + by*i + cxy*i*j;
    }

    if (pa)
        *pa = a;
    if (pbx)
        *pbx = bx;
    if (pby)
        *pby = by;
    if (pcxy)
        *pcxy = cxy;

    return field;
}

static GwyField*
make_random_noise_field(void)
{
    GwyField *field = make_random_planar_field(NULL, NULL, NULL);
    gint n = gwy_field_get_xres(field)*gwy_field_get_yres(field);
    gdouble *d = gwy_field_get_data(field);
    gdouble s = exp(2*g_test_rand_double() - 1);
    for (gint k = 0; k < n; k++)
        d[k] = s*(0.1*d[k] + g_test_rand_double());

    return field;
}

static GwyField*
cut_smaller_field(GwyField *field, gint *pcol, gint *prow)
{
    gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
    gint smallxres = (xres <= 2 ? xres : g_test_rand_int_range(2, xres + 1));
    gint smallyres = (yres <= 2 ? yres : g_test_rand_int_range(2, yres + 1));
    gint col = g_test_rand_int_range(0, xres+1 - smallxres);
    gint row = g_test_rand_int_range(0, yres+1 - smallyres);
    GwyField *small = gwy_field_area_extract(field, col, row, smallxres, smallyres);
    if (pcol)
        *pcol = col;
    if (prow)
        *prow = row;
    return small;
}

void
test_field_stats_avg_planar(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble expected = a + bx*(xres - 1.0)/2.0 + by*(yres - 1.0)/2.0;
        g_assert_cmpfloat_with_epsilon(gwy_field_get_avg(field), expected, 4*DBL_EPSILON*xres*yres);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_avg_bilinear(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by, cxy;
        GwyField *field = make_random_bilinear_field(&a, &bx, &by, &cxy);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint xm1 = xres - 1, ym1 = yres - 1;
        gdouble expected = a + bx*xm1/2.0 + by*ym1/2.0 + cxy*xm1*ym1/4.0;
        g_assert_cmpfloat_with_epsilon(gwy_field_get_avg(field), expected, 8*DBL_EPSILON*yres*yres);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_avg_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint col, row;
        GwyField *small = cut_smaller_field(field, &col, &row);
        gint width = gwy_field_get_xres(small), height = gwy_field_get_yres(small);
        g_assert_cmpfloat_with_epsilon(gwy_NIELD_area_avg(field, NULL, GWY_MASK_IGNORE, col, row, width, height),
                                       gwy_field_get_avg(small),
                                       2*DBL_EPSILON);
        g_assert_finalize_object(small);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_min_planar(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble expected = fmin(fmin(a, a + bx*(xres - 1)), fmin(a + by*(yres - 1), a + bx*(xres - 1) + by*(yres - 1)));
        g_assert_cmpfloat_with_epsilon(gwy_field_get_min(field), expected, 2*DBL_EPSILON*xres*yres);

        gwy_field_invalidate(field);
        gdouble min, max;
        gwy_field_get_min_max(field, &min, &max);
        g_assert_cmpfloat_with_epsilon(min, expected, 2*DBL_EPSILON*xres*yres);

        g_assert_finalize_object(field);
    }
}

void
test_field_stats_min_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint col, row;
        GwyField *small = cut_smaller_field(field, &col, &row);
        gint width = gwy_field_get_xres(small), height = gwy_field_get_yres(small);
        g_assert_cmpfloat_with_epsilon(gwy_NIELD_area_min(field, NULL, GWY_MASK_IGNORE, col, row, width, height),
                                       gwy_field_get_min(small),
                                       2*DBL_EPSILON);
        g_assert_finalize_object(small);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_max_planar(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble expected = fmax(fmax(a, a + bx*(xres - 1)), fmax(a + by*(yres - 1), a + bx*(xres - 1) + by*(yres - 1)));
        g_assert_cmpfloat_with_epsilon(gwy_field_get_max(field), expected, 2*DBL_EPSILON*xres*yres);

        gwy_field_invalidate(field);
        gdouble min, max;
        gwy_field_get_min_max(field, &min, &max);
        g_assert_cmpfloat_with_epsilon(max, expected, 2*DBL_EPSILON*xres*yres);

        g_assert_finalize_object(field);
    }
}

void
test_field_stats_max_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint col, row;
        GwyField *small = cut_smaller_field(field, &col, &row);
        gint width = gwy_field_get_xres(small), height = gwy_field_get_yres(small);
        g_assert_cmpfloat_with_epsilon(gwy_NIELD_area_max(field, NULL, GWY_MASK_IGNORE, col, row, width, height),
                                       gwy_field_get_max(small),
                                       2*DBL_EPSILON);
        g_assert_finalize_object(small);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_min_max_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        gdouble min = gwy_NIELD_area_min(field, NULL, GWY_MASK_IGNORE, col, row, width, height);
        gwy_field_invalidate(field);
        gdouble max = gwy_NIELD_area_max(field, NULL, GWY_MASK_IGNORE, col, row, width, height);
        gwy_field_invalidate(field);
        gdouble mmin, mmax;
        gwy_NIELD_area_min_max(field, NULL, GWY_MASK_IGNORE, col, row, width, height, &mmin, &mmax);
        g_assert_cmpfloat_with_epsilon(min, mmin, 2*DBL_EPSILON);
        g_assert_cmpfloat_with_epsilon(max, mmax, 2*DBL_EPSILON);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_min_max_masked(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        GwyNield *mask = make_random_mask(xres, yres, TRUE, 3);
        GwyMaskingType masking;
        if (g_test_rand_int_range(0, 2))
            masking = GWY_MASK_INCLUDE;
        else
            masking = GWY_MASK_EXCLUDE;
        gdouble min = gwy_NIELD_area_min(field, mask, masking, col, row, width, height);
        gwy_field_invalidate(field);
        gdouble max = gwy_NIELD_area_max(field, mask, masking, col, row, width, height);
        gwy_field_invalidate(field);
        gdouble mmin, mmax;
        gwy_NIELD_area_min_max(field, mask, masking, col, row, width, height, &mmin, &mmax);
        g_assert_cmpfloat_with_epsilon(min, mmin, 2*DBL_EPSILON);
        g_assert_cmpfloat_with_epsilon(max, mmax, 2*DBL_EPSILON);
        g_assert_finalize_object(mask);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_median_full(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint npts = gwy_field_get_xres(field)*gwy_field_get_yres(field);
        gdouble *d = g_new(gdouble, npts);
        gwy_assign(d, gwy_field_get_data_const(field), npts);
        qsort(d, npts, sizeof(gdouble), gwy_compare_double);
        gdouble expected = d[npts/2];
        g_assert_cmpfloat(gwy_field_get_median(field), ==, expected);
        g_free(d);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_median_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint col, row;
        GwyField *small = cut_smaller_field(field, &col, &row);
        gint width = gwy_field_get_xres(small), height = gwy_field_get_yres(small);
        g_assert_cmpfloat(gwy_NIELD_area_median(field, NULL, GWY_MASK_IGNORE, col, row, width, height),
                          ==,
                          gwy_field_get_median(small));
        g_assert_finalize_object(small);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_sum_planar(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble expected = xres*yres*(a + bx*(xres - 1.0)/2.0 + by*(yres - 1.0)/2.0);
        g_assert_cmpfloat_with_epsilon(gwy_field_get_sum(field), expected, 2*DBL_EPSILON*xres*xres*yres*yres);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_sum_bilinear(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by, cxy;
        GwyField *field = make_random_bilinear_field(&a, &bx, &by, &cxy);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint xm1 = xres - 1, ym1 = yres - 1;
        gdouble expected = xres*yres*(a + bx*xm1/2.0 + by*ym1/2.0 + cxy*xm1*ym1/4.0);
        g_assert_cmpfloat_with_epsilon(gwy_field_get_sum(field), expected, 4*DBL_EPSILON*xres*xres*yres*yres);
        g_assert_finalize_object(field);
    }
}

/* FIXME: With CLANG fast math, even a simple summation like here can give differences of the order 100ε, not a few ε.
 * Is it too much? It behaves more like accumulating bad roundings, instead of them cancelling out on average and the
 * error growing only with √N. */
void
test_field_stats_sum_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint col, row;
        GwyField *small = cut_smaller_field(field, &col, &row);
        gint width = gwy_field_get_xres(small), height = gwy_field_get_yres(small);
        g_assert_cmpfloat_with_epsilon(gwy_NIELD_area_sum(field, NULL, GWY_MASK_IGNORE, col, row, width, height),
                                       gwy_field_get_sum(small),
                                       4*DBL_EPSILON);
        g_assert_finalize_object(small);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_mean_square_planar(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint xm1 = xres - 1, ym1 = yres - 1;
        gdouble x6 = (2*xres - 1.0)/6.0, y6 = (2*yres - 1.0)/6.0;
        gdouble expected = (a*a + bx*bx*x6*xm1 + by*by*y6*ym1 + a*bx*xm1 + a*by*ym1 + bx*by*0.5*xm1*ym1);
        g_assert_cmpfloat_with_epsilon(gwy_field_get_mean_square(field), expected,
                                       4*DBL_EPSILON*xres*xres*yres*yres);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_mean_square_bilinear(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by, cxy;
        GwyField *field = make_random_bilinear_field(&a, &bx, &by, &cxy);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint xm1 = xres - 1, ym1 = yres - 1;
        gdouble x6 = (2*xres - 1.0)/6.0, y6 = (2*yres - 1.0)/6.0;
        gdouble expected = (a*a + bx*bx*xm1*x6 + by*by*ym1*y6 + cxy*cxy*xm1*x6*ym1*y6
                            + a*bx*xm1 + a*by*ym1 + (bx*by + a*cxy)*0.5*xm1*ym1
                            + bx*cxy*xm1*x6*ym1 + by*cxy*xm1*ym1*y6);
        g_assert_cmpfloat_with_epsilon(gwy_field_get_mean_square(field), expected, 4*DBL_EPSILON*xres*xres*yres*yres);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_mean_square_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint col, row;
        GwyField *small = cut_smaller_field(field, &col, &row);
        gint width = gwy_field_get_xres(small), height = gwy_field_get_yres(small);
        g_assert_cmpfloat_with_epsilon(gwy_NIELD_area_mean_square(field, NULL, GWY_MASK_IGNORE,
                                                                  col, row, width, height),
                                       gwy_field_get_mean_square(small),
                                       2*DBL_EPSILON);
        g_assert_finalize_object(small);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_rms_planar(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble bx, by;
        GwyField *field = make_random_planar_field(NULL, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint npix = xres*yres;
        gint x2m1 = xres*xres - 1, y2m1 = yres*yres - 1;
        gdouble expected = sqrt(bx*bx*x2m1/12.0 + by*by*y2m1/12.0);
        expected *= sqrt(npix/(npix - 1.0));   // Degrees of freedom
        g_assert_cmpfloat_with_epsilon(gwy_field_get_rms(field), expected, 2*DBL_EPSILON*npix);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_rms_bilinear(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble bx, by, cxy;
        GwyField *field = make_random_bilinear_field(NULL, &bx, &by, &cxy);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint npix = xres*yres;
        gint xm1 = xres - 1, ym1 = yres - 1;
        gint x2m1 = xres*xres - 1, y2m1 = yres*yres - 1;
        gdouble expected = sqrt(bx*bx*x2m1/12.0 + by*by*y2m1/12.0
                                + bx*cxy*x2m1*ym1/12.0 + by*cxy*xm1*y2m1/12.0
                                + cxy*cxy*xm1*ym1*(7.0*xres*yres + xres + yres - 5.0)/144.0);
        expected *= sqrt(npix/(npix - 1.0));   // Degrees of freedom
        g_assert_cmpfloat_with_epsilon(gwy_field_get_rms(field), expected, 2*DBL_EPSILON*npix);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_rms_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint col, row;
        GwyField *small = cut_smaller_field(field, &col, &row);
        gint width = gwy_field_get_xres(small), height = gwy_field_get_yres(small);
        g_assert_cmpfloat_with_epsilon(gwy_NIELD_area_rms(field, NULL, GWY_MASK_IGNORE, col, row, width, height),
                                       gwy_field_get_rms(small),
                                       2*DBL_EPSILON);
        g_assert_finalize_object(small);
        g_assert_finalize_object(field);
    }
}

// There is no (sane) explicit formula for Ra because z - mean(z) changes sign along an arbitary straight line.
// So we are again doing a summation, except using an explicit formula for values, not reading any pixel data.
static gdouble
expected_ra_for_planar_field(gint xres, gint yres, gdouble bx, gdouble by)
{
    gdouble s = 0.0, cj = 0.5*(xres - 1), ci = 0.5*(yres - 1);
    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            s += fabs((i - ci)*by + (j - cj)*bx);
    }
    gint n = xres*yres;
    gdouble corr = pow(n/(n - 1.0), 0.62);
    return n > 1 ? s/n*corr : 0;
}

void
test_field_stats_get_stats_planar(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint npix = xres*yres;
        gint xm1 = xres - 1, ym1 = yres - 1;
        gint x2m1 = xres*xres - 1, y2m1 = yres*yres - 1;
        gdouble tx = bx*bx*x2m1, ty = by*by*y2m1;
        gdouble expected_avg = a + bx*xm1/2.0 + by*ym1/2.0;
        gdouble m2 = (tx + ty)/12.0;
        gdouble expected_rms = sqrt(m2) * sqrt(npix/(npix - 1.0)); // Degrees of freedom
        gdouble expected_ra = expected_ra_for_planar_field(xres, yres, bx, by);
        gdouble m4 = (bx*bx*tx*(xres*xres - 7.0/3.0) + by*by*ty*(yres*yres - 7.0/3.0))/80.0 + tx*ty/24.0;
        gdouble expected_skew = 0.0;
        gdouble expected_kurtosis = m4/(m2*m2) - 3.0;
        gdouble avg, rms, ra, skew, kurtosis;
        gwy_field_get_stats(field, &avg, &ra, &rms, &skew, &kurtosis);
        g_assert_cmpfloat_with_epsilon(avg, expected_avg, 4*DBL_EPSILON*xres*yres);
        g_assert_cmpfloat_with_epsilon(rms, expected_rms, 4*DBL_EPSILON*xres*yres);
        g_assert_cmpfloat_with_epsilon(ra, expected_ra, 4*DBL_EPSILON*xres*yres);
        g_assert_cmpfloat_with_epsilon(skew, expected_skew, 30*DBL_EPSILON*xres*yres);
        g_assert_cmpfloat_with_epsilon(kurtosis, expected_kurtosis, 50*DBL_EPSILON*xres*yres);

        // This should sucessfully do nothing.
        gwy_field_get_stats(field, NULL, NULL, NULL, NULL, NULL);

        gwy_field_invalidate(field);
        gwy_field_get_stats(field, &avg, NULL, NULL, NULL, NULL);
        g_assert_cmpfloat_with_epsilon(avg, expected_avg, 4*DBL_EPSILON*xres*yres);
        gwy_field_invalidate(field);
        gwy_field_get_stats(field, NULL, &ra, NULL, NULL, NULL);
        g_assert_cmpfloat_with_epsilon(ra, expected_ra, 4*DBL_EPSILON*xres*yres);
        gwy_field_invalidate(field);
        gwy_field_get_stats(field, NULL, NULL, &rms, NULL, NULL);
        g_assert_cmpfloat_with_epsilon(rms, expected_rms, 4*DBL_EPSILON*xres*yres);
        gwy_field_invalidate(field);
        gwy_field_get_stats(field, NULL, NULL, NULL, &skew, NULL);
        g_assert_cmpfloat_with_epsilon(skew, expected_skew, 30*DBL_EPSILON*xres*yres);
        gwy_field_invalidate(field);
        gwy_field_get_stats(field, NULL, NULL, NULL, NULL, &kurtosis);
        g_assert_cmpfloat_with_epsilon(kurtosis, expected_kurtosis, 50*DBL_EPSILON*xres*yres);

        g_assert_finalize_object(field);
    }
}

// There is no (sane) explicit formula for Ra because z - mean(z) changes sign along an arbitary straight line.
// So we are again doing a summation, except using an explicit formula for values, not reading any pixel data.
static gdouble
expected_ra_for_bilinear_field(gint xres, gint yres, gdouble bx, gdouble by, gdouble cxy)
{
    gdouble s = 0.0, cj = 0.5*(xres - 1), ci = 0.5*(yres - 1);
    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            s += fabs((i - ci)*by + (j - cj)*bx + (i*j - ci*cj)*cxy);
    }
    gint n = xres*yres;
    gdouble corr = pow(n/(n - 1.0), 0.62);
    return n > 1 ? s/n*corr : 0;
}

void
test_field_stats_get_stats_bilinear(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by, cxy;
        GwyField *field = make_random_bilinear_field(&a, &bx, &by, &cxy);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint npix = xres*yres;
        gint xm1 = xres - 1, ym1 = yres - 1;
        gint x2m1 = xres*xres - 1, y2m1 = yres*yres - 1;
        gdouble expected_avg = a + bx*xm1/2.0 + by*ym1/2.0 + cxy*xm1*ym1/4.0;
        gdouble m2 = (bx*bx*x2m1/12.0 + by*by*y2m1/12.0
                      + bx*cxy*x2m1*ym1/12.0 + by*cxy*xm1*y2m1/12.0
                      + cxy*cxy*xm1*ym1*(7.0*xres*yres + xres + yres - 5.0)/144.0);
        gdouble m3 = cxy*x2m1*y2m1*(bx*cxy*xm1/2.0 + by*cxy*ym1/2.0 + bx*by + cxy*cxy*xm1*ym1/4.0)/24.0;
        /*
        gdouble m4b = (bx*bx*bx*bx*x2m1*(3.0*xres*xres - 7.0)/240.0
                       + bx*bx*by*by*x2m1*y2m1/24.0
                       + by*by*by*by*y2m1*(3.0*yres*yres - 7.0)/240.0);
        gdouble m4x = bx*cxy*(bx*bx*x2m1*(3.0*xres*xres - 7.0)*ym1
                              + bx*cxy*x2m1*ym1*(17.0*xres*xres*yres - 10.0*xres*yres - xres*xres
                                                 - 23.0*yres - 10.0*xres + 19.0)/4.0
                              + cxy*cxy*x2m1*ym1*ym1*(11.0*xres*xres*yres - 10.0*xres*yres + 5*xres*xres
                                                      - 9.0*yres - 10.0*xres + 5.0)/4.0)/120.0;
        gdouble m4y = by*cxy*(by*by*y2m1*(3.0*yres*yres - 7.0)*ym1
                              + by*cxy*y2m1*xm1*(17.0*yres*yres*xres - 10.0*yres*xres - yres*yres
                                                 - 23.0*xres - 10.0*yres + 19.0)/4.0
                              + cxy*cxy*y2m1*xm1*xm1*(11.0*yres*yres*xres - 10.0*yres*xres + 5*yres*yres
                                                      - 9.0*xres - 10.0*yres + 5.0)/4.0)/120.0;
        gdouble m4c = cxy*cxy*cxy*cxy*xm1*ym1*(429.0*xres*xres*xres*yres*yres*yres
                                               - 231.0*xres*xres*xres*yres*yres - 231.0*xres*xres*yres*yres*yres
                                               - 441.0*xres*xres*xres*yres
                                               - 291.0*xres*xres*yres*yres
                                               - 441.0*xres*yres*yres*yres
                                               - 141.0*xres*xres*xres + 699.0*xres*xres*yres
                                               + 699.0*xres*yres*yres - 141.0*yres*yres*yres
                                               + 399.0*xres*xres - 11.0*xres*yres + 399.0*yres*yres
                                               - 311.0*yres - 311.0*xres - 11.0)/57600.0;
        gdouble m4 = m4b + m4x + m4y + m4c;
        */
        gdouble expected_rms = sqrt(m2) * sqrt(npix/(npix - 1.0)); // Degrees of freedom
        gdouble expected_skew = m3/sqrt(m2*m2*m2);
        //gdouble expected_kurtosis = m4/(m2*m2);
        gdouble expected_ra = expected_ra_for_bilinear_field(xres, yres, bx, by, cxy);
        gdouble avg, rms, ra, skew, kurtosis;
        gwy_field_get_stats(field, &avg, &ra, &rms, &skew, &kurtosis);
        g_assert_cmpfloat_with_epsilon(avg, expected_avg, 4*DBL_EPSILON*xres*yres);
        g_assert_cmpfloat_with_epsilon(rms, expected_rms, 4*DBL_EPSILON*xres*yres);
        g_assert_cmpfloat_with_epsilon(ra, expected_ra, 2*DBL_EPSILON*xres*yres);
        g_assert_cmpfloat_with_epsilon(skew, expected_skew, 20*DBL_EPSILON*xres*yres);
        //g_assert_cmpfloat_with_epsilon(kurtosis, expected_kurtosis, 50*DBL_EPSILON*xres*yres);

        // This should sucessfully do nothing.
        gwy_field_get_stats(field, NULL, NULL, NULL, NULL, NULL);

        gwy_field_invalidate(field);
        gwy_field_get_stats(field, &avg, NULL, NULL, NULL, NULL);
        g_assert_cmpfloat_with_epsilon(avg, expected_avg, 4*DBL_EPSILON*xres*yres);
        gwy_field_invalidate(field);
        gwy_field_get_stats(field, NULL, &ra, NULL, NULL, NULL);
        g_assert_cmpfloat_with_epsilon(ra, expected_ra, 4*DBL_EPSILON*xres*yres);
        gwy_field_invalidate(field);
        gwy_field_get_stats(field, NULL, NULL, &rms, NULL, NULL);
        g_assert_cmpfloat_with_epsilon(rms, expected_rms, 4*DBL_EPSILON*xres*yres);
        gwy_field_invalidate(field);
        gwy_field_get_stats(field, NULL, NULL, NULL, &skew, NULL);
        g_assert_cmpfloat_with_epsilon(skew, expected_skew, 20*DBL_EPSILON*xres*yres);
        gwy_field_invalidate(field);
        //gwy_field_get_stats(field, NULL, NULL, NULL, NULL, &kurtosis);
        //g_assert_cmpfloat_with_epsilon(kurtosis, expected_kurtosis, 50*DBL_EPSILON*xres*yres);

        g_assert_finalize_object(field);
    }
}

void
test_field_stats_get_stats_area(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint col, row;
        GwyField *small = cut_smaller_field(field, &col, &row);
        gint width = gwy_field_get_xres(small), height = gwy_field_get_yres(small);
        gdouble avg, rms, ra, skew, kurtosis;
        gwy_NIELD_area_stats(field, NULL, GWY_MASK_IGNORE, col, row, width, height,
                             &avg, &ra, &rms, &skew, &kurtosis);
        gdouble small_avg, small_rms, small_ra, small_skew, small_kurtosis;
        gwy_field_get_stats(small, &small_avg, &small_ra, &small_rms, &small_skew, &small_kurtosis);
        g_assert_cmpfloat_with_epsilon(avg, small_avg, 2*DBL_EPSILON);
        g_assert_cmpfloat_with_epsilon(ra, small_ra, 2*DBL_EPSILON);
        g_assert_cmpfloat_with_epsilon(rms, small_rms, 2*DBL_EPSILON);
        /* XXX: These tolerances were eyeballed and may be too small. */
        g_assert_cmpfloat_with_epsilon(skew, small_skew, 6*DBL_EPSILON);
        g_assert_cmpfloat_with_epsilon(kurtosis, small_kurtosis, 10*DBL_EPSILON);
        g_assert_finalize_object(small);
        g_assert_finalize_object(field);
    }
}

static void
test_field_stats_area_common(gboolean make_pixels_square)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble dx = gwy_field_get_dx(field), dy = gwy_field_get_dy(field);
        gdouble dA = dx*dy;
        if (make_pixels_square) {
            dx = dy = sqrt(dA);
            gwy_field_set_xreal(field, dx*xres);
            gwy_field_set_yreal(field, dy*yres);
        }
        gdouble rbx = bx/dx, rby = by/dy; // plane level uses per-pixels slopes!
        // The calculation uses mirror boundary extrapolation, meaning image edges are flatter than inside.
        // It is one of possible definitions, but check that it does what we expect.
        gdouble expected_inner = (xres - 1)*(yres - 1)*dA*sqrt(1 + rbx*rbx + rby*rby);
        gdouble expected_hedges = (xres - 1)*dA*sqrt(1 + rbx*rbx);
        gdouble expected_vedges = (yres - 1)*dA*sqrt(1 + rby*rby);
        gdouble expected_corners = dA;
        gdouble expected = expected_inner + expected_hedges + expected_vedges + expected_corners;
        g_assert_cmpfloat_with_epsilon(gwy_field_get_surface_area(field), expected,
                                       DBL_EPSILON*xres*yres*expected);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_area_square(void)
{
    test_field_stats_area_common(TRUE);
}

void
test_field_stats_area_nonsquare(void)
{
    test_field_stats_area_common(FALSE);
}

void
test_field_stats_variation_common(gboolean make_pixels_square)
{
    guint n = g_test_slow() ? 10000 : 1000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble dx = gwy_field_get_dx(field), dy = gwy_field_get_dy(field);
        gdouble dA = dx*dy;
        if (make_pixels_square) {
            dx = dy = sqrt(dA);
            gwy_field_set_xreal(field, dx*xres);
            gwy_field_set_yreal(field, dy*yres);
        }
        gdouble rbx = bx/dx, rby = by/dy; // plane level uses per-pixels slopes!
        // The calculation uses mirror boundary extrapolation, meaning image edges are flatter than inside.
        // It is one of possible definitions, but check that it does what we expect.
        gdouble expected_inner = (xres - 1)*(yres - 1)*dA*sqrt(rbx*rbx + rby*rby);
        gdouble expected_hedges = (xres - 1)*dA*fabs(rbx);
        gdouble expected_vedges = (yres - 1)*dA*fabs(rby);
        gdouble expected_corners = 0.0;
        gdouble expected = expected_inner + expected_hedges + expected_vedges + expected_corners;
        g_assert_cmpfloat_with_epsilon(gwy_field_get_variation(field), expected,
                                       4*DBL_EPSILON*xres*yres*expected);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_variation_square(void)
{
    test_field_stats_variation_common(TRUE);
}

void
test_field_stats_variation_nonsquare(void)
{
    test_field_stats_variation_common(FALSE);
}

void
test_field_stats_volume_constant(void)
{
    static GwyFieldVolumeMethod methods[] = {
        GWY_FIELD_VOLUME_DEFAULT,
        GWY_FIELD_VOLUME_GWYDDION2,
        GWY_FIELD_VOLUME_TRIANGULAR,
        GWY_FIELD_VOLUME_BILINEAR,
        GWY_FIELD_VOLUME_BIQUADRATIC,
    };
    guint n = 100;

    for (guint i = 0; i < n; i++) {
        gint xres = g_test_rand_int_range(3, 20);
        gint yres = g_test_rand_int_range(3, 20);
        gdouble dx = 0.5 + g_test_rand_double();
        gdouble dy = 0.5 + g_test_rand_double();
        gdouble c = 2*g_test_rand_double() - 1.0;
        GwyField *field = gwy_field_new(xres, yres, xres*dx, yres*dy, FALSE);
        gwy_field_fill(field, c);
        GwyNield *mask = make_random_mask(xres, yres, FALSE, 1);
        guint npix = gwy_nield_count(mask);

        for (guint k = 0; k < G_N_ELEMENTS(methods); k++) {
            gdouble expected = dx*dy*c*npix;
            gdouble volume = gwy_NIELD_area_volume(field, mask, GWY_MASK_INCLUDE, methods[k], 0, 0, xres, yres);
            // The error seems to get pretty large. Not sure why as there should be no cancellation whatsoever.
            g_assert_cmpfloat_with_epsilon(volume, expected, 60*xres*yres*DBL_EPSILON);

            expected = dx*dy*c*(xres*yres - npix);
            volume = gwy_NIELD_area_volume(field, mask, GWY_MASK_EXCLUDE, methods[k], 0, 0, xres, yres);
            g_assert_cmpfloat_with_epsilon(volume, expected, 60*xres*yres*DBL_EPSILON);
        }
        g_assert_finalize_object(mask);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_volume_planar(void)
{
    static GwyFieldVolumeMethod methods[] = {
        GWY_FIELD_VOLUME_DEFAULT,
        GWY_FIELD_VOLUME_GWYDDION2,
        GWY_FIELD_VOLUME_TRIANGULAR,
        GWY_FIELD_VOLUME_BILINEAR,
        GWY_FIELD_VOLUME_BIQUADRATIC,
    };
    guint n = g_test_slow() ? 1000 : 10000;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        if (xres < 3 || yres < 3)
            continue;

        gdouble dx = gwy_field_get_dx(field), dy = gwy_field_get_dy(field);
        gdouble zavg = a + bx*(xres - 1.0)/2.0 + by*(yres - 1.0)/2.0;
        gdouble expected = (xres - 2)*dx * (yres - 2)*dy * zavg;
        for (guint k = 0; k < G_N_ELEMENTS(methods); k++) {
            gdouble volume = gwy_NIELD_area_volume(field, NULL, GWY_MASK_IGNORE, methods[k], 1, 1, xres-2, yres-2);
            g_assert_cmpfloat_with_epsilon(volume, expected, 5e3*(fabs(expected) + 1)*DBL_EPSILON);
        }
        g_assert_finalize_object(field);
    }
}

static void
test_field_stats_masked_common(MaskedStatFunc calc_stat, ConstantDataFunc const_value)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint i = 0; i < n; i++) {
        GwyField *field = make_random_noise_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        GwyNield *nield = make_random_mask(xres, yres, TRUE, 3);

        gdouble ref_incl = calc_stat(field, nield, GWY_MASK_INCLUDE, 0, 0, xres, yres);
        gdouble ref_excl = calc_stat(field, nield, GWY_MASK_EXCLUDE, 0, 0, xres, yres);
        gdouble ref_area_incl = calc_stat(field, nield, GWY_MASK_INCLUDE, col, row, width, height);
        gdouble ref_area_excl = calc_stat(field, nield, GWY_MASK_EXCLUDE, col, row, width, height);

        gdouble a = g_test_rand_double(), b = g_test_rand_double();

        /* Test that we get the known value from the constant-filled regions and the complement is unaffected.
         * This does not test whether the reported value is actually right; only that it is calculated consistently
         * with respect to masking.
         *
         * g_assert_cmpfloat_with_epsilon() has a strict inequality so we have to fudge it to get 0 == 0 succeed.
         * Also some of the expected values can be G_MAXDOUBLE (when n = 0), so be careful with multiplications. */
        gint a_nfull = gwy_nield_area_count(nield, NULL, GWY_MASK_IGNORE, 0, 0, xres, yres);
        gint a_narea = gwy_nield_area_count(nield, NULL, GWY_MASK_IGNORE, col, row, width, height);
        GwyField *repl_a = gwy_field_copy(field);
        gwy_NIELD_area_fill(repl_a, nield, GWY_MASK_INCLUDE, 0, 0, xres, yres, a);
        gdouble full_a = const_value(a, a_nfull), area_a = const_value(a, a_narea);
        gdouble a_incl = calc_stat(repl_a, nield, GWY_MASK_INCLUDE, 0, 0, xres, yres);
        gdouble a_excl = calc_stat(repl_a, nield, GWY_MASK_EXCLUDE, 0, 0, xres, yres);
        gdouble a_area_incl = calc_stat(repl_a, nield, GWY_MASK_INCLUDE, col, row, width, height);
        gdouble a_area_excl = calc_stat(repl_a, nield, GWY_MASK_EXCLUDE, col, row, width, height);
        g_assert_cmpfloat_with_epsilon(a_incl, full_a, fabs(full_a)*DBL_EPSILON*20 + 20*DBL_EPSILON);
        g_assert_cmpfloat(a_excl, ==, ref_excl);
        g_assert_cmpfloat_with_epsilon(a_area_incl, area_a, fabs(area_a)*DBL_EPSILON*20 + 20*DBL_EPSILON);
        g_assert_cmpfloat(a_area_excl, ==, ref_area_excl);
        g_assert_finalize_object(repl_a);

        gint b_nfull = xres*yres - a_nfull, b_narea = width*height - a_narea;
        GwyField *repl_b = gwy_field_copy(field);
        gwy_NIELD_area_fill(repl_b, nield, GWY_MASK_EXCLUDE, 0, 0, xres, yres, b);
        gdouble full_b = const_value(b, b_nfull), area_b = const_value(b, b_narea);
        gdouble b_incl = calc_stat(repl_b, nield, GWY_MASK_INCLUDE, 0, 0, xres, yres);
        gdouble b_excl = calc_stat(repl_b, nield, GWY_MASK_EXCLUDE, 0, 0, xres, yres);
        gdouble b_area_incl = calc_stat(repl_b, nield, GWY_MASK_INCLUDE, col, row, width, height);
        gdouble b_area_excl = calc_stat(repl_b, nield, GWY_MASK_EXCLUDE, col, row, width, height);
        g_assert_cmpfloat(b_incl, ==, ref_incl);
        g_assert_cmpfloat_with_epsilon(b_excl, full_b, fabs(full_b)*DBL_EPSILON*20 + 20*DBL_EPSILON);
        g_assert_cmpfloat(b_area_incl, ==, ref_area_incl);
        g_assert_cmpfloat_with_epsilon(b_area_excl, area_b, fabs(area_b)*DBL_EPSILON*20 + 20*DBL_EPSILON);
        g_assert_finalize_object(repl_b);

        g_assert_finalize_object(nield);
        g_assert_finalize_object(field);
    }
}

static gdouble
min_for_const(gdouble c, gsize n)
{
    return n ? c : G_MAXDOUBLE;
}

void
test_field_stats_min_masked(void)
{
    test_field_stats_masked_common(gwy_NIELD_area_min, min_for_const);
}

static gdouble
max_for_const(gdouble c, gsize n)
{
    return n ? c : -G_MAXDOUBLE;
}

void
test_field_stats_max_masked(void)
{
    test_field_stats_masked_common(gwy_NIELD_area_max, max_for_const);
}

static gdouble
avg_for_const(gdouble c, gsize n)
{
    return n ? c : 0.0;
}

void
test_field_stats_avg_masked(void)
{
    test_field_stats_masked_common(gwy_NIELD_area_avg, avg_for_const);
}

static gdouble
sum_for_const(gdouble c, gsize n)
{
    return n*c;
}

void
test_field_stats_sum_masked(void)
{
    test_field_stats_masked_common(gwy_NIELD_area_sum, sum_for_const);
}

static gdouble
rms_for_const(G_GNUC_UNUSED gdouble c, G_GNUC_UNUSED gsize n)
{
    return 0.0;
}

void
test_field_stats_rms_masked(void)
{
    test_field_stats_masked_common(gwy_NIELD_area_rms, rms_for_const);
}

static gdouble
mean_square_for_const(gdouble c, gsize n)
{
    return n ? c*c : 0.0;
}

void
test_field_stats_mean_square_masked(void)
{
    test_field_stats_masked_common(gwy_NIELD_area_mean_square, mean_square_for_const);
}

static gdouble
median_for_const(gdouble c, gsize n)
{
    return n ? c : 0.0;
}

void
test_field_stats_median_masked(void)
{
    test_field_stats_masked_common(gwy_NIELD_area_median, median_for_const);
}

void
test_field_fit_plane_random(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble fita, fitbx, fitby;
        gwy_field_fit_plane(field, &fita, &fitbx, &fitby);
        gdouble s = fmax(fabs(a), fmax(fabs(bx), fabs(by)));
        g_assert_cmpfloat_with_epsilon(fita, a, 4*DBL_EPSILON*xres*yres*s);
        g_assert_cmpfloat_with_epsilon(fitbx, bx, DBL_EPSILON*xres*yres*s);
        g_assert_cmpfloat_with_epsilon(fitby, by, DBL_EPSILON*xres*yres*s);
        gwy_field_plane_level(field, a, bx, by);
        g_assert_cmpfloat(fabs(gwy_field_get_min(field)), <, DBL_EPSILON*xres*yres);
        g_assert_cmpfloat(fabs(gwy_field_get_max(field)), <, DBL_EPSILON*xres*yres);
        g_assert_finalize_object(field);
    }
}

void
test_field_fit_plane_degenerate(void)
{
    gdouble fita, fitbx, fitby;

    const double data1[3] = { 3.0, 2.0, 1.0 };
    GwyField *field1 = gwy_field_new(3, 1, 1.0, 1.0, FALSE);
    gwy_assign(gwy_field_get_data(field1), data1, 3);
    gwy_field_fit_plane(field1, &fita, &fitbx, &fitby);
    g_assert_cmpfloat_with_epsilon(fita, 3.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, -1.0, 6*DBL_EPSILON);
    g_assert_cmpfloat(fitby, ==, 0.0);
    g_assert_finalize_object(field1);

    const double data2[2] = { -1.0, 1.0 };
    GwyField *field2 = gwy_field_new(1, 2, 1.0, 1.0, FALSE);
    gwy_assign(gwy_field_get_data(field2), data2, 2);
    gwy_field_fit_plane(field2, &fita, &fitbx, &fitby);
    g_assert_cmpfloat_with_epsilon(fita, -1.0, 4*DBL_EPSILON);
    g_assert_cmpfloat(fitbx, ==, 0.0);
    g_assert_cmpfloat_with_epsilon(fitby, 2.0, 4*DBL_EPSILON);
    g_assert_finalize_object(field2);
}

void
test_field_fit_plane_area_basic(void)
{
    gdouble fita, fitbx, fitby;
    gint rank;
    const gdouble data[12] = {
        7.5, 7.5, 7.5, 7.5,
        7.5, 7.5, 3.1, 4.1,
        7.5, 7.5, 1.1, 4.1,
    };

    GwyField *field = gwy_field_new(4, 3, 1.0, 1.0, FALSE);
    gwy_assign(gwy_field_get_data(field), data, 12);

    /* Here we mainly check that the constant coefficient is with respect to the area's top left corner. */
    rank = gwy_NIELD_area_fit_plane(field, NULL, GWY_MASK_IGNORE, 2, 1, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 3);
    g_assert_cmpfloat_with_epsilon(fita, 2.6, 10*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, 2.0, 10*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, -1.0, 10*DBL_EPSILON);

    GwyNield *mask = gwy_nield_new(4, 3);
    gwy_nield_fill(mask, 1);
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_INCLUDE, 2, 1, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 3);
    g_assert_cmpfloat_with_epsilon(fita, 2.6, 10*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, 2.0, 10*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, -1.0, 10*DBL_EPSILON);

    gwy_nield_set_val(mask, 3, 2, 0);
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_INCLUDE, 2, 1, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 3);
    g_assert_cmpfloat_with_epsilon(fita, 3.1, 10*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, 1.0, 10*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, -2.0, 10*DBL_EPSILON);

    g_assert_finalize_object(mask);
    g_assert_finalize_object(field);
}

void
test_field_fit_plane_area_degenerate_2x2(void)
{
    gdouble fita, fitbx, fitby;
    gint rank;
    const gdouble data[4] = {
        3.0,  1.0,
        -1.0, 2.0,
    };

    GwyField *field = gwy_field_new(2, 2, 1.0, 1.0, FALSE);
    gwy_assign(gwy_field_get_data(field), data, 4);

    GwyNield *mask = gwy_nield_new(2, 2);

    /* Column 0 */
    gwy_nield_clear(mask);
    gwy_nield_set_val(mask, 0, 0, 1);
    gwy_nield_set_val(mask, 0, 1, 1);
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_INCLUDE, 0, 0, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 2);
    g_assert_cmpfloat_with_epsilon(fita, 3.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, 0.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, -4.0, 6*DBL_EPSILON);

    /* Column 1 */
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_EXCLUDE, 0, 0, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 2);
    g_assert_cmpfloat_with_epsilon(fita, 1.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, 0.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, 1.0, 6*DBL_EPSILON);

    /* Row 0 */
    gwy_nield_clear(mask);
    gwy_nield_set_val(mask, 0, 1, 1);
    gwy_nield_set_val(mask, 1, 1, 1);
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_EXCLUDE, 0, 0, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 2);
    g_assert_cmpfloat_with_epsilon(fita, 3.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, -2.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, 0.0, 6*DBL_EPSILON);

    /* Row 1 */
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_INCLUDE, 0, 0, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 2);
    g_assert_cmpfloat_with_epsilon(fita, -1.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, 3.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, 0.0, 6*DBL_EPSILON);

    /* Major diagonal */
    gwy_nield_clear(mask);
    gwy_nield_set_val(mask, 0, 0, 1);
    gwy_nield_set_val(mask, 1, 1, 1);
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_INCLUDE, 0, 0, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 2);
    g_assert_cmpfloat_with_epsilon(fita, 3.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, -0.5, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, -0.5, 6*DBL_EPSILON);

    /* Minor diagonal (the constant is for area's top left corner!) */
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_EXCLUDE, 0, 0, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 2);
    g_assert_cmpfloat_with_epsilon(fita, 0.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitbx, 1.0, 6*DBL_EPSILON);
    g_assert_cmpfloat_with_epsilon(fitby, -1.0, 6*DBL_EPSILON);

    /* Single pixel */
    gwy_nield_clear(mask);
    gwy_nield_set_val(mask, 0, 1, 1);
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_INCLUDE, 0, 0, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 1);
    g_assert_cmpfloat_with_epsilon(fita, -1.0, 6*DBL_EPSILON);
    g_assert_cmpfloat(fitbx, ==, 0.0);
    g_assert_cmpfloat(fitby, ==, 0.0);

    /* Nothing */
    gwy_nield_clear(mask);
    rank = gwy_NIELD_area_fit_plane(field, mask, GWY_MASK_INCLUDE, 0, 0, 2, 2, &fita, &fitbx, &fitby);
    g_assert_cmpint(rank, ==, 0);
    g_assert_cmpfloat(fita, ==, 0.0);
    g_assert_cmpfloat(fitbx, ==, 0.0);
    g_assert_cmpfloat(fitby, ==, 0.0);

    g_assert_finalize_object(mask);
    g_assert_finalize_object(field);
}

void
test_field_fit_poly_linear(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by;
        GwyField *field = make_random_planar_field(&a, &bx, &by);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint term_powers[6] = { 0, 0, 1, 0, 0, 1 };
        gdouble coeffs[3];
        gwy_field_fit_poly(field, NULL, GWY_MASK_IGNORE, 3, term_powers, coeffs);
        gwy_field_subtract_poly(field, 3, term_powers, coeffs);
        g_assert_cmpfloat(fabs(gwy_field_get_min(field)), <, 4*DBL_EPSILON*xres*yres);
        g_assert_cmpfloat(fabs(gwy_field_get_max(field)), <, 4*DBL_EPSILON*xres*yres);
        g_assert_finalize_object(field);
    }
}

void
test_field_fit_poly_masked(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    guint count = 0;
    for (guint i = 0; i < n; i++) {
        gdouble a, bx, by, cxy;
        GwyField *field = make_random_bilinear_field(&a, &bx, &by, &cxy);
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        GwyNield *nield = make_random_mask(xres, yres, TRUE, 1);
        /* Ignore too small pixel sets. */
        if (gwy_nield_area_count(nield, NULL, GWY_MASK_IGNORE, col, row, width, height) < 20)
            continue;

        gint term_powers[8] = { 0, 0, 1, 0, 0, 1, 1, 1 };
        gdouble coeffs[4];
        gwy_NIELD_area_fit_poly(field, nield, GWY_MASK_INCLUDE, col, row, width, height, 4, term_powers, coeffs);
        /* The coefficients are meant for the area, not the entire field. This is just how the function is defined. */
        GwyField *part = gwy_field_area_extract(field, col, row, width, height);
        gwy_field_subtract_poly(part, 4, term_powers, coeffs);
        /* The errors can be pretty large because we are using only a bit of the data and extrapolate. */
        g_assert_cmpfloat(fabs(gwy_field_get_min(part)), <, 1e-12*width*height);
        g_assert_cmpfloat(fabs(gwy_field_get_max(part)), <, 1e-12*width*height);
        g_assert_finalize_object(part);
        g_assert_finalize_object(nield);
        g_assert_finalize_object(field);
        count++;
    }

    g_assert_cmpuint(count, >=, n/5);
}

GwyField*
make_ij_field(void)
{
    gint xres = g_test_rand_int_range(2, 20);
    gint yres = g_test_rand_int_range(2, 20);
    GwyField *field = gwy_field_new(xres, yres, xres, yres, FALSE);
    gdouble *d = gwy_field_get_data(field);

    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            d[i*xres + j] = SOME_NUMBER*i + j;
    }

    return field;
}

void
test_field_area_extract(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyField *field = make_ij_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width), row = g_test_rand_int_range(0, yres-height);
        GwyField *extracted = gwy_field_area_extract(field, col, row, width, height);
        const gdouble *d = gwy_field_get_data_const(extracted);
        for (gint i = 0; i < height; i++) {
            for (gint j = 0; j < width; j++) {
                g_assert_cmpfloat(d[i*width + j], ==, SOME_NUMBER*(i + row) + (j + col));
            }
        }
        g_assert_finalize_object(extracted);
        g_assert_finalize_object(field);
    }
}

void
test_field_crop(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyField *field = make_ij_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width), row = g_test_rand_int_range(0, yres-height);
        gwy_field_crop(field, col, row, width, height);
        const gdouble *d = gwy_field_get_data_const(field);
        for (gint i = 0; i < height; i++) {
            for (gint j = 0; j < width; j++) {
                g_assert_cmpfloat(d[i*width + j], ==, SOME_NUMBER*(i + row) + (j + col));
            }
        }
        g_assert_finalize_object(field);
    }
}

void
test_field_area_copy(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyField *field = make_ij_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        gint xres2 = g_test_rand_int_range(width, 25), yres2 = g_test_rand_int_range(height, 25);
        gint destcol = g_test_rand_int_range(0, xres2-width+1), destrow = g_test_rand_int_range(0, yres2-height+1);
        GwyField *destfield = gwy_field_new(xres2, yres2, xres2, yres2, FALSE);
        gwy_field_fill(destfield, G_PI);
        gwy_field_area_copy(field, destfield, col, row, width, height, destcol, destrow);
        const gdouble *d = gwy_field_get_data_const(destfield);
        for (gint i = 0; i < yres2; i++) {
            for (gint j = 0; j < xres2; j++) {
                if (i >= destrow && i-destrow < height && j >= destcol && j-destcol < width)
                    g_assert_cmpfloat(d[i*xres2 + j], ==, SOME_NUMBER*(i + row - destrow) + (j + col - destcol));
                else
                    g_assert_cmpfloat(d[i*xres2 + j], ==, G_PI);
            }
        }
        g_assert_finalize_object(destfield);
        g_assert_finalize_object(field);
    }
}

void
test_field_flip_x(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyField *field = make_ij_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gwy_field_flip(field, TRUE, FALSE);
        const gdouble *d = gwy_field_get_data_const(field);
        for (gint i = 0; i < yres; i++) {
            for (gint j = 0; j < xres; j++) {
                gint flipped_j = xres-1 - j;
                g_assert_cmpfloat(d[i*xres + j], ==, SOME_NUMBER*i + flipped_j);
            }
        }
        g_assert_finalize_object(field);
    }
}

void
test_field_flip_y(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyField *field = make_ij_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gwy_field_flip(field, FALSE, TRUE);
        const gdouble *d = gwy_field_get_data_const(field);
        for (gint i = 0; i < yres; i++) {
            gint flipped_i = yres-1 - i;
            for (gint j = 0; j < xres; j++) {
                g_assert_cmpfloat(d[i*xres + j], ==, SOME_NUMBER*flipped_i + j);
            }
        }
        g_assert_finalize_object(field);
    }
}

void
test_field_flip_xy(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyField *field = make_ij_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gwy_field_flip(field, TRUE, TRUE);
        const gdouble *d = gwy_field_get_data_const(field);
        for (gint i = 0; i < yres; i++) {
            gint flipped_i = yres-1 - i;
            for (gint j = 0; j < xres; j++) {
                gint flipped_j = xres-1 - j;
                g_assert_cmpfloat(d[i*xres + j], ==, SOME_NUMBER*flipped_i + flipped_j);
            }
        }
        g_assert_finalize_object(field);
    }
}

void
test_field_setval(void)
{
    gint xres = g_test_rand_int_range(2, 20);
    gint yres = g_test_rand_int_range(2, 20);
    GwyField *field = gwy_field_new(xres, yres, xres, yres, FALSE);
    gwy_field_fill(field, G_PI);

    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            gwy_field_set_val(field, j, i, i*SOME_NUMBER + j);
    }
    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            g_assert_cmpfloat(gwy_field_get_val(field, j, i), ==, i*SOME_NUMBER + j);
    }
    g_assert_finalize_object(field);
}

void
test_field_transpose_major(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyField *field = make_ij_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble xreal = gwy_field_get_xreal(field), yreal = gwy_field_get_yreal(field);
        GwyField *transposed = gwy_field_new(1, 1, 1.0, 1.0, FALSE);
        gwy_field_transpose(field, transposed, FALSE);
        g_assert_cmpint(gwy_field_get_xres(transposed), ==, yres);
        g_assert_cmpint(gwy_field_get_yres(transposed), ==, xres);
        g_assert_cmpint(gwy_field_get_xreal(transposed), ==, yreal);
        g_assert_cmpint(gwy_field_get_yreal(transposed), ==, xreal);
        const gdouble *d = gwy_field_get_data_const(transposed);
        for (gint i = 0; i < xres; i++) {
            for (gint j = 0; j < yres; j++)
                g_assert_cmpfloat(d[i*yres + j], ==, SOME_NUMBER*j + i);
        }
        GwyField *transposed2 = gwy_field_new(1, 1, 1.0, 1.0, FALSE);
        gwy_field_transpose(transposed, transposed2, FALSE);
        field_assert_equal(G_OBJECT(field), G_OBJECT(transposed2));
        g_assert_finalize_object(transposed2);
        g_assert_finalize_object(transposed);
        g_assert_finalize_object(field);
    }
}

void
test_field_transpose_minor(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyField *field = make_ij_field();
        gint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
        gdouble xreal = gwy_field_get_xreal(field), yreal = gwy_field_get_yreal(field);
        GwyField *transposed = gwy_field_new(1, 1, 1.0, 1.0, FALSE);
        gwy_field_transpose(field, transposed, TRUE);
        g_assert_cmpint(gwy_field_get_xres(transposed), ==, yres);
        g_assert_cmpint(gwy_field_get_yres(transposed), ==, xres);
        g_assert_cmpint(gwy_field_get_xreal(transposed), ==, yreal);
        g_assert_cmpint(gwy_field_get_yreal(transposed), ==, xreal);
        const gdouble *d = gwy_field_get_data_const(transposed);
        for (gint i = 0; i < xres; i++) {
            for (gint j = 0; j < yres; j++)
                g_assert_cmpfloat(d[i*yres + j], ==, SOME_NUMBER*(yres-1 - j) + (xres-1 - i));
        }
        GwyField *transposed2 = gwy_field_new(1, 1, 1.0, 1.0, FALSE);
        gwy_field_transpose(transposed, transposed2, TRUE);
        field_assert_equal(G_OBJECT(field), G_OBJECT(transposed2));
        g_assert_finalize_object(transposed2);
        g_assert_finalize_object(transposed);
        g_assert_finalize_object(field);
    }
}

static void
assert_field_compatibility(gint xres1, gint yres1,
                           gdouble xreal1, gdouble yreal1,
                           const gchar *xyunit1, const gchar *zunit1,
                           gint xres2, gint yres2,
                           gdouble xreal2, gdouble yreal2,
                           const gchar *xyunit2, const gchar *zunit2,
                           GwyDataMismatchFlags flags_to_test,
                           GwyDataMismatchFlags expected_result)
{
    GwyField *field1 = gwy_field_new(xres1, yres1, xreal1, yreal1, TRUE);
    if (xyunit1)
        gwy_unit_set_from_string(gwy_field_get_unit_xy(field1), xyunit1);
    if (zunit1)
        gwy_unit_set_from_string(gwy_field_get_unit_z(field1), zunit1);

    GwyField *field2 = gwy_field_new(xres2, yres2, xreal2, yreal2, TRUE);
    if (xyunit2)
        gwy_unit_set_from_string(gwy_field_get_unit_xy(field2), xyunit2);
    if (zunit2)
        gwy_unit_set_from_string(gwy_field_get_unit_z(field2), zunit2);

    g_assert_cmphex(gwy_field_is_incompatible(field1, field2, flags_to_test), ==, expected_result);
    g_assert_finalize_object(field1);
    g_assert_finalize_object(field2);
}

void
test_field_compatibility_res(void)
{
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               1, 2, 0.3, 1e40, NULL, "A",
                               GWY_DATA_MISMATCH_RES, 0);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               1, 1, 0.3, 1e40, NULL, "A",
                               GWY_DATA_MISMATCH_RES, GWY_DATA_MISMATCH_RES);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               2, 2, 0.3, 1e40, NULL, "A",
                               GWY_DATA_MISMATCH_RES, GWY_DATA_MISMATCH_RES);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               2, 1, 0.3, 1e40, NULL, "A",
                               GWY_DATA_MISMATCH_RES, GWY_DATA_MISMATCH_RES);
}

void
test_field_compatibility_real(void)
{
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               3, 4, G_PI, 7.0, NULL, "A",
                               GWY_DATA_MISMATCH_REAL, 0);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               3, 4, 0.99*G_PI, 7.0, NULL, "A",
                               GWY_DATA_MISMATCH_REAL, GWY_DATA_MISMATCH_REAL);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               3, 4, G_PI, 7.01, NULL, "A",
                               GWY_DATA_MISMATCH_REAL, GWY_DATA_MISMATCH_REAL);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               3, 4, 7.0, G_PI, NULL, "A",
                               GWY_DATA_MISMATCH_REAL, GWY_DATA_MISMATCH_REAL);
    /* Tiny differences do not break compatibility. */
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               3, 4, G_PI*(1.0 - 1e-14), 7.0*(1.0 + 1e-14), NULL, "A",
                               GWY_DATA_MISMATCH_REAL, 0);
}

void
test_field_compatibility_lateral(void)
{
    assert_field_compatibility(1, 2, G_PI, 7.0, NULL, NULL,
                               3, 4, 1.0, 1e30, NULL, NULL,
                               GWY_DATA_MISMATCH_LATERAL, 0);
    assert_field_compatibility(1, 2, G_PI, 7.0, NULL, "m",
                               3, 4, 1.0, 1e30, "", "A",
                               GWY_DATA_MISMATCH_LATERAL, 0);
    assert_field_compatibility(1, 2, G_PI, 7.0, "mV", "m",
                               3, 4, 1.0, 1e30, "kilovolts", "A",
                               GWY_DATA_MISMATCH_LATERAL, 0);
    assert_field_compatibility(1, 2, G_PI, 7.0, "V", "m",
                               3, 4, 1.0, 1e30, "A", "V",
                               GWY_DATA_MISMATCH_LATERAL, GWY_DATA_MISMATCH_LATERAL);
    assert_field_compatibility(1, 2, G_PI, 7.0, "N", "",
                               3, 4, 1.0, 1e30, "", "N",
                               GWY_DATA_MISMATCH_LATERAL, GWY_DATA_MISMATCH_LATERAL);
}

void
test_field_compatibility_value(void)
{
    assert_field_compatibility(1, 2, G_PI, 7.0, NULL, NULL,
                               3, 4, 1.0, 1e30, NULL, NULL,
                               GWY_DATA_MISMATCH_VALUE, 0);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", NULL,
                               3, 4, 1.0, 1e30, "A", "",
                               GWY_DATA_MISMATCH_VALUE, 0);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", "micrometres",
                               3, 4, 1.0, 1e30, "A", "pm",
                               GWY_DATA_MISMATCH_VALUE, 0);
    assert_field_compatibility(1, 2, G_PI, 7.0, "m", "A",
                               3, 4, 1.0, 1e30, "A", "V",
                               GWY_DATA_MISMATCH_VALUE, GWY_DATA_MISMATCH_VALUE);
    assert_field_compatibility(1, 2, G_PI, 7.0, "", "N",
                               3, 4, 1.0, 1e30, "N", "",
                               GWY_DATA_MISMATCH_VALUE, GWY_DATA_MISMATCH_VALUE);
}

void
test_field_compatibility_measure(void)
{
    assert_field_compatibility(1, 2, 2e-9, 1e-9, "m", "g",
                               3, 4, 6e-9, 2e-9, "A", "N",
                               GWY_DATA_MISMATCH_MEASURE, 0);
    assert_field_compatibility(1, 2, 2e-9, 1e-9, "m", "g",
                               3, 4, 6.01e-9, 2e-9, "A", "N",
                               GWY_DATA_MISMATCH_MEASURE, GWY_DATA_MISMATCH_MEASURE);
    assert_field_compatibility(1, 2, 2e-9, 1e-9, "m", "g",
                               3, 4, 6e-9, 1.99e-9, "A", "N",
                               GWY_DATA_MISMATCH_MEASURE, GWY_DATA_MISMATCH_MEASURE);
    assert_field_compatibility(1, 2, 2e-9, 1e-9, "m", "g",
                               3, 4, 1.5e-9, 1e-9, "A", "N",
                               GWY_DATA_MISMATCH_MEASURE, GWY_DATA_MISMATCH_MEASURE);
    assert_field_compatibility(1, 2, 2e-9, 1e-9, "m", "g",
                               3, 4, 2e-9, 1e-9, "A", "N",
                               GWY_DATA_MISMATCH_MEASURE, GWY_DATA_MISMATCH_MEASURE);
    /* Tiny differences do not break compatibility. */
    assert_field_compatibility(1, 2, 2e-9, 1e-9, "m", "g",
                               3, 4, 6e-9*(1.0 - 1e-14), 2e-9*(1.0 + 1e-14), "A", "N",
                               GWY_DATA_MISMATCH_MEASURE, 0);
}

void
test_field_compatibility_mixed(void)
{
    GwyDataMismatchFlags all_flags = (GWY_DATA_MISMATCH_RES
                                      | GWY_DATA_MISMATCH_REAL
                                      | GWY_DATA_MISMATCH_MEASURE
                                      | GWY_DATA_MISMATCH_LATERAL
                                      | GWY_DATA_MISMATCH_VALUE);
    assert_field_compatibility(1, 2, 2e-9, 1e-9, "m", "V",
                               3, 4, 2e-9, 1e-9, "A", "V",
                               all_flags,
                               GWY_DATA_MISMATCH_RES | GWY_DATA_MISMATCH_MEASURE | GWY_DATA_MISMATCH_LATERAL);
    assert_field_compatibility(3, 4, 3e-9, 4e-9, "m", "V",
                               3, 4, 4e-9, 3e-9, "A", "N",
                               all_flags,
                               GWY_DATA_MISMATCH_REAL | GWY_DATA_MISMATCH_MEASURE
                               | GWY_DATA_MISMATCH_LATERAL | GWY_DATA_MISMATCH_VALUE);
    assert_field_compatibility(3, 4, 3e-9, 40e-9, NULL, NULL,
                               6, 8, 6e-9, 80e-9, NULL, NULL,
                               all_flags,
                               GWY_DATA_MISMATCH_RES | GWY_DATA_MISMATCH_REAL);
}

void
test_field_clear_area(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(3, 15);
        gint yres = g_test_rand_int_range(3, 15);
        GwyField *field = gwy_field_new(xres, yres, xres, yres, FALSE);

        gdouble a = 10.0*g_test_rand_double() - 5.0;
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        gdouble avg, expected;

        gwy_field_fill(field, a);
        gwy_field_invalidate(field);
        g_assert_cmpfloat(gwy_field_get_min(field), ==, a);
        g_assert_cmpfloat(gwy_field_get_max(field), ==, a);
        gwy_field_invalidate(field);

        gwy_field_area_clear(field, col, row, width, height);
        avg = gwy_field_get_avg(field);
        expected = (xres*yres - width*height)*a/(xres*yres);
        g_assert_cmpfloat_with_epsilon(avg, expected, xres*yres*fabs(a)*DBL_EPSILON);

        GwyField *detail1 = gwy_field_area_extract(field, col, row, width, height);
        avg = gwy_field_get_avg(detail1);
        expected = 0.0;
        g_assert_cmpfloat(avg, ==, expected);
        g_assert_finalize_object(detail1);

        g_assert_finalize_object(field);
    }
}

void
test_field_fill_area(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(3, 15);
        gint yres = g_test_rand_int_range(3, 15);
        GwyField *field = gwy_field_new(xres, yres, xres, yres, FALSE);

        gdouble a = g_test_rand_double(), b = g_test_rand_double();
        gwy_field_fill(field, a);
        gwy_field_invalidate(field);
        g_assert_cmpfloat(gwy_field_get_min(field), ==, a);
        g_assert_cmpfloat(gwy_field_get_max(field), ==, a);
        gwy_field_invalidate(field);

        GwyNield *nield = gwy_nield_new(xres, yres);
        gint width = g_test_rand_int_range(2, xres), height = g_test_rand_int_range(2, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        guint m = 0, npix = g_test_rand_int_range(1, width*height);
        gdouble avg, expected;

        while (m < npix) {
            gint i = row + g_test_rand_int_range(0, height);
            gint j = col + g_test_rand_int_range(0, width);
            if (!gwy_nield_get_val(nield, j, i)) {
                gwy_nield_set_val(nield, j, i, 1);
                m++;
            }
        }

        gwy_NIELD_area_fill(field, nield, GWY_MASK_INCLUDE, col, row, width, height, b);
        g_assert_cmpfloat(gwy_field_get_min(field), ==, fmin(a, b));
        g_assert_cmpfloat(gwy_field_get_max(field), ==, fmax(a, b));

        avg = gwy_field_get_avg(field);
        expected = (npix*b + (xres*yres - npix)*a)/(xres*yres);
        g_assert_cmpfloat_with_epsilon(avg, expected, xres*yres*fmax(a, b)*DBL_EPSILON);

        GwyField *detail1 = gwy_field_area_extract(field, col, row, width, height);
        avg = gwy_field_get_avg(detail1);
        expected = (npix*b + (width*height - npix)*a)/(width*height);
        g_assert_cmpfloat_with_epsilon(avg, expected, width*height*fmax(a, b)*DBL_EPSILON);
        g_assert_finalize_object(detail1);

        gwy_NIELD_area_fill(field, nield, GWY_MASK_IGNORE, col, row, width, height, b);
        avg = gwy_field_get_avg(field);
        expected = (width*height*b + (xres*yres - width*height)*a)/(xres*yres);
        g_assert_cmpfloat_with_epsilon(avg, expected, xres*yres*fmax(a, b)*DBL_EPSILON);

        gwy_NIELD_area_fill(field, nield, GWY_MASK_IGNORE, col, row, width, height, a);
        avg = gwy_field_get_avg(field);
        expected = a;
        g_assert_cmpfloat_with_epsilon(avg, expected, xres*yres*a*DBL_EPSILON);

        gwy_NIELD_area_fill(field, nield, GWY_MASK_EXCLUDE, col, row, width, height, b);
        avg = gwy_field_get_avg(field);
        expected = ((width*height - npix)*b + (xres*yres - (width*height - npix))*a)/(xres*yres);
        g_assert_cmpfloat_with_epsilon(avg, expected, xres*yres*fmax(a, b)*DBL_EPSILON);

        GwyField *detail2 = gwy_field_area_extract(field, col, row, width, height);
        avg = gwy_field_get_avg(detail2);
        expected = (npix*a + (width*height - npix)*b)/(width*height);
        g_assert_cmpfloat_with_epsilon(avg, expected, width*height*fmax(a, b)*DBL_EPSILON);
        g_assert_finalize_object(detail2);

        gwy_NIELD_area_fill(field, nield, GWY_MASK_INCLUDE, col, row, width, height, b);
        avg = gwy_field_get_avg(field);
        expected = (width*height*b + (xres*yres - width*height)*a)/(xres*yres);
        g_assert_cmpfloat_with_epsilon(avg, expected, xres*yres*fmax(a, b)*DBL_EPSILON);

        gwy_field_fill(field, a);
        gwy_nield_fill(nield, 4);
        gwy_NIELD_area_fill(field, nield, GWY_MASK_INCLUDE, col, row, width, height, b);
        avg = gwy_field_get_avg(field);
        expected = (width*height*b + (xres*yres - width*height)*a)/(xres*yres);
        g_assert_cmpfloat_with_epsilon(avg, expected, xres*yres*fmax(a, b)*DBL_EPSILON);

        g_assert_finalize_object(nield);
        g_assert_finalize_object(field);
    }
}

void
test_field_fill_area_mask(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(3, 15);
        gint yres = g_test_rand_int_range(3, 15);
        GwyField *field = gwy_field_new(xres, yres, xres, yres, FALSE);
        gint width = g_test_rand_int_range(2, xres), height = g_test_rand_int_range(2, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);

        gdouble a = g_test_rand_double(), b = g_test_rand_double(), c = g_test_rand_double();
        gwy_field_fill(field, c);
        gwy_field_invalidate(field);

        GwyNield *nield = make_random_mask(xres, yres, TRUE, 3);
        gwy_field_area_fill_mask(field, nield, col, row, width, height, a, b);
        gint count = gwy_nield_area_count(nield, NULL, GWY_MASK_IGNORE, col, row, width, height);

        if (count) {
            g_assert_cmpfloat(gwy_NIELD_area_min(field, nield, GWY_MASK_INCLUDE, col, row, width, height), ==, a);
            g_assert_cmpfloat(gwy_NIELD_area_max(field, nield, GWY_MASK_INCLUDE, col, row, width, height), ==, a);
        }
        if (width*height - count) {
            g_assert_cmpfloat(gwy_NIELD_area_min(field, nield, GWY_MASK_EXCLUDE, col, row, width, height), ==, b);
            g_assert_cmpfloat(gwy_NIELD_area_max(field, nield, GWY_MASK_EXCLUDE, col, row, width, height), ==, b);
        }

        if (width*height < xres*yres) {
            gwy_nield_fill(nield, 1);
            gwy_nield_area_clear(nield, col, row, width, height);
            g_assert_cmpfloat(gwy_NIELD_area_min(field, nield, GWY_MASK_INCLUDE, 0, 0, xres, yres), ==, c);
            g_assert_cmpfloat(gwy_NIELD_area_max(field, nield, GWY_MASK_INCLUDE, 0, 0, xres, yres), ==, c);
        }

        g_assert_finalize_object(nield);
        g_assert_finalize_object(field);
    }
}

static void
test_field_extend_simple(const gdouble *bigdata, guint bigxres, guint bigyres,
                         guint xres, guint yres, guint col, guint row,
                         GwyExteriorType exterior, gdouble fill_value,
                         gdouble eps)
{
    GwyField *bigfield = gwy_field_new(bigxres, bigyres, bigxres, bigyres, FALSE);
    gwy_assign(gwy_field_get_data(bigfield), bigdata, bigxres*bigyres);

    GwyField *field = gwy_field_area_extract(bigfield, col, row, xres, yres);
    for (gint left = 0; left <= col; left++) {
        for (gint right = 0; right <= bigxres - (col + xres); right++) {
            for (gint up = 0; up <= row; up++) {
                for (gint down = 0; down <= bigyres - (row + yres); down++) {
                    //g_printerr("ext [%d,%d]x[%d,%d]\n", left, right, up, down);
                    GwyField *expected = gwy_field_area_extract(bigfield,
                                                                col - left, row - up,
                                                                xres + left + right, yres + up + down);
                    gwy_field_set_xoffset(expected, -left);
                    gwy_field_set_yoffset(expected, -up);

                    GwyField *extended = gwy_field_extend(field, left, right, up, down,
                                                          exterior, fill_value, TRUE);

                    if (eps == 0.0)
                        field_assert_equal(G_OBJECT(extended), G_OBJECT(expected));
                    else {
                        assert_field_properties(extended, expected);
                        assert_field_content(extended, gwy_field_get_data(expected),
                                             gwy_field_get_xres(expected)*gwy_field_get_yres(expected),
                                             1e-14);
                    }

                    g_assert_finalize_object(extended);
                    g_assert_finalize_object(expected);
                }
            }
        }
    }

    g_assert_finalize_object(field);
    g_assert_finalize_object(bigfield);
}

void
test_field_extend_border(void)
{
    enum { bigxres = 9, bigyres = 9 };
    enum { xres = 3, yres = 2 };
    enum { col = 3, row = 4 };
    const gdouble bigdata[bigxres*bigyres] = {
        4.0,  4.0,  4.0,    4.0,  -1.0, 0.0,   0.0, 0.0, 0.0,
        4.0,  4.0,  4.0,    4.0,  -1.0, 0.0,   0.0, 0.0, 0.0,
        4.0,  4.0,  4.0,    4.0,  -1.0, 0.0,   0.0, 0.0, 0.0,
        4.0,  4.0,  4.0,    4.0,  -1.0, 0.0,   0.0, 0.0, 0.0,

        4.0,  4.0,  4.0,    4.0,  -1.0, 0.0,   0.0, 0.0, 0.0,
        G_PI, G_PI, G_PI,   G_PI, 5.5,  1e5,   1e5, 1e5, 1e5,

        G_PI, G_PI, G_PI,   G_PI, 5.5,  1e5,   1e5, 1e5, 1e5,
        G_PI, G_PI, G_PI,   G_PI, 5.5,  1e5,   1e5, 1e5, 1e5,
        G_PI, G_PI, G_PI,   G_PI, 5.5,  1e5,   1e5, 1e5, 1e5,
    };

    test_field_extend_simple(bigdata, bigxres, bigyres, xres, yres, col, row, GWY_EXTERIOR_BORDER, G_MAXDOUBLE, 0.0);
}

void
test_field_extend_mirror(void)
{
    enum { bigxres = 11, bigyres = 8 };
    enum { xres = 3, yres = 2 };
    enum { col = 4, row = 3 };
    const gdouble bigdata[bigxres*bigyres] = {
        1e5, 1e5, 5.5,  G_PI,   G_PI, 5.5,  1e5,   1e5, 5.5,  G_PI, G_PI,
        1e5, 1e5, 5.5,  G_PI,   G_PI, 5.5,  1e5,   1e5, 5.5,  G_PI, G_PI,
        0.0, 0.0, -1.0, 4.0,    4.0,  -1.0, 0.0,   0.0, -1.0, 4.0,  4.0,

        0.0, 0.0, -1.0, 4.0,    4.0,  -1.0, 0.0,   0.0, -1.0, 4.0,  4.0,
        1e5, 1e5, 5.5,  G_PI,   G_PI, 5.5,  1e5,   1e5, 5.5,  G_PI, G_PI,

        1e5, 1e5, 5.5,  G_PI,   G_PI, 5.5,  1e5,   1e5, 5.5,  G_PI, G_PI,
        0.0, 0.0, -1.0, 4.0,    4.0,  -1.0, 0.0,   0.0, -1.0, 4.0,  4.0,
        0.0, 0.0, -1.0, 4.0,    4.0,  -1.0, 0.0,   0.0, -1.0, 4.0,  4.0,
    };

    test_field_extend_simple(bigdata, bigxres, bigyres, xres, yres, col, row, GWY_EXTERIOR_MIRROR, G_MAXDOUBLE, 0.0);
}

void
test_field_extend_periodic(void)
{
    enum { bigxres = 11, bigyres = 8 };
    enum { xres = 3, yres = 2 };
    enum { col = 4, row = 3 };
    const gdouble bigdata[bigxres*bigyres] = {
        1e5, G_PI, 5.5,  1e5,   G_PI, 5.5,  1e5,   G_PI, 5.5,  1e5, G_PI,
        0.0, 4.0,  -1.0, 0.0,   4.0,  -1.0, 0.0,   4.0,  -1.0, 0.0, 4.0,
        1e5, G_PI, 5.5,  1e5,   G_PI, 5.5,  1e5,   G_PI, 5.5,  1e5, G_PI,

        0.0, 4.0,  -1.0, 0.0,   4.0,  -1.0, 0.0,   4.0,  -1.0, 0.0, 4.0,
        1e5, G_PI, 5.5,  1e5,   G_PI, 5.5,  1e5,   G_PI, 5.5,  1e5, G_PI,

        0.0, 4.0,  -1.0, 0.0,   4.0,  -1.0, 0.0,   4.0,  -1.0, 0.0, 4.0,
        1e5, G_PI, 5.5,  1e5,   G_PI, 5.5,  1e5,   G_PI, 5.5,  1e5, G_PI,
        0.0, 4.0,  -1.0, 0.0,   4.0,  -1.0, 0.0,   4.0,  -1.0, 0.0, 4.0,
    };

    test_field_extend_simple(bigdata, bigxres, bigyres, xres, yres, col, row, GWY_EXTERIOR_PERIODIC, G_MAXDOUBLE, 0.0);
}

void
test_field_extend_fixed(void)
{
    enum { bigxres = 11, bigyres = 8 };
    enum { xres = 3, yres = 2 };
    enum { col = 4, row = 3 };
    const gdouble bigdata[bigxres*bigyres] = {
        G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,
        G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,
        G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,

        G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,   4.0,     -1.0,    0.0,       G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,
        G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,   G_PI,    5.5,     1e5,       G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,

        G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,
        G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,
        G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2,   G_SQRT2, G_SQRT2, G_SQRT2, G_SQRT2,
    };

    test_field_extend_simple(bigdata, bigxres, bigyres, xres, yres, col, row, GWY_EXTERIOR_FIXED_VALUE, G_SQRT2, 0.0);
}

/* Verify reproduction of constant with some tolerance. */
void
test_field_extend_laplace(void)
{
    enum { bigxres = 11, bigyres = 8 };
    enum { xres = 3, yres = 2 };
    enum { col = 4, row = 3 };
    gdouble bigdata[bigxres*bigyres];

    for (guint i = 0; i < bigxres*bigyres; i++)
        bigdata[i] = GWY_SQRT_PI;

    test_field_extend_simple(bigdata, bigxres, bigyres, xres, yres, col, row, GWY_EXTERIOR_LAPLACE, G_MAXDOUBLE, 1e-14);
}

/* Verify reproduction of constant with some tolerance. */
void
test_field_extend_connect(void)
{
    enum { bigxres = 11, bigyres = 8 };
    enum { xres = 3, yres = 2 };
    enum { col = 4, row = 3 };
    gdouble bigdata[bigxres*bigyres];

    for (guint i = 0; i < bigxres*bigyres; i++)
        bigdata[i] = GWY_SQRT_PI;

    test_field_extend_simple(bigdata, bigxres, bigyres, xres, yres, col, row, GWY_EXTERIOR_CONNECT, G_MAXDOUBLE, 1e-14);
}

static void
assert_dist_moments(GwyLine *dist, gdouble avg, gdouble rms, gdouble eps)
{
    /* Compare the first and second moment computed from the distribution to the real values. Of course, they will not
     * be the same, but it should not be too far. */
    gint nstat = gwy_line_get_res(dist);
    gdouble dx = gwy_line_get_dx(dist), xoff = gwy_line_get_offset(dist);
    const gdouble *distdata = gwy_line_get_data_const(dist);

    gdouble s1 = 0.0;
    for (gint i = 0; i < nstat; i++) {
        gdouble z = ((i + 0.5)*dx + xoff);
        s1 += distdata[i]*z*dx;
    }

    g_assert_cmpfloat_with_epsilon(s1, avg, eps);

    gdouble s2 = 0.0;
    for (gint i = 0; i < nstat; i++) {
        gdouble z = ((i + 0.5)*dx + xoff);
        s2 += distdata[i]*(z - avg)*(z - avg)*dx;
    }
    s2 = sqrt(s2);
    g_assert_cmpfloat_with_epsilon(s2, rms, eps);
}

void
test_field_stats_height_dist_full(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(10, 30), yres = g_test_rand_int_range(10, 30);
        GwyField *field = gwy_field_new(xres, yres, xres, yres, FALSE);
        gdouble a = 5.0*g_test_rand_double() + 3.0, b = 1.5*g_test_rand_double();

        for (gint i = 0; i < yres; i++) {
            gdouble y = (i + 0.5)/yres;
            for (gint j = 0; j < xres; j++) {
                gdouble x = (j + 0.5)/xres;
                gwy_field_set_val(field, j, i, a + exp(b*(x + y)) + g_test_rand_double());
            }
        }

        GwyLine *dist = gwy_line_new(1, 1.0, FALSE);
        gwy_NIELD_area_height_dist(field, NULL, GWY_MASK_IGNORE, dist, 0, 0, xres, yres, -1);

        assert_dist_moments(dist, gwy_field_get_avg(field), gwy_field_get_rms(field), 0.1);

        g_assert_finalize_object(dist);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_height_dist_area(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(15, 30), yres = g_test_rand_int_range(15, 30);
        gint width = g_test_rand_int_range(10, xres), height = g_test_rand_int_range(10, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        GwyField *field = gwy_field_new(xres, yres, xres, yres, FALSE);
        gdouble a = 5.0*g_test_rand_double() + 3.0, b = 1.5*g_test_rand_double();
        gwy_field_fill(field, G_MAXDOUBLE);

        for (gint i = 0; i < height; i++) {
            gdouble y = (i + 0.5)/height;
            for (gint j = 0; j < width; j++) {
                gdouble x = (j + 0.5)/width;
                gwy_field_set_val(field, j + col, i + row, a + exp(b*(x + y)) + g_test_rand_double());
            }
        }

        GwyLine *dist = gwy_line_new(1, 1.0, FALSE);
        gwy_NIELD_area_height_dist(field, NULL, GWY_MASK_IGNORE, dist, col, row, width, height, -1);

        assert_dist_moments(dist,
                            gwy_NIELD_area_avg(field, NULL, GWY_MASK_IGNORE, col, row, width, height),
                            gwy_NIELD_area_rms(field, NULL, GWY_MASK_IGNORE, col, row, width, height),
                            0.1);

        g_assert_finalize_object(dist);
        g_assert_finalize_object(field);
    }
}

void
test_field_stats_height_dist_masked(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(17, 40), yres = g_test_rand_int_range(17, 40);
        gint width = g_test_rand_int_range(15, xres), height = g_test_rand_int_range(15, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        GwyField *field = gwy_field_new(xres, yres, xres, yres, FALSE);
        gdouble a = 5.0*g_test_rand_double() + 3.0, b = 1.5*g_test_rand_double();
        gwy_field_fill(field, G_MAXDOUBLE);

        GwyNield *nield = make_random_mask(xres, yres, TRUE, 3);

        for (gint i = 0; i < height; i++) {
            gdouble y = (i + 0.5)/height;
            for (gint j = 0; j < width; j++) {
                gdouble x = (j + 0.5)/width;
                if (gwy_nield_get_val(nield, j + col, i + row) > 0)
                    gwy_field_set_val(field, j + col, i + row, a + exp(b*(x + y)) + g_test_rand_double());
            }
        }

        GwyLine *dist = gwy_line_new(1, 1.0, FALSE);
        gwy_NIELD_area_height_dist(field, nield, GWY_MASK_INCLUDE, dist, col, row, width, height, -1);

        assert_dist_moments(dist,
                            gwy_NIELD_area_avg(field, nield, GWY_MASK_INCLUDE, col, row, width, height),
                            gwy_NIELD_area_rms(field, nield, GWY_MASK_INCLUDE, col, row, width, height),
                            0.1);

        g_assert_finalize_object(nield);
        g_assert_finalize_object(dist);
        g_assert_finalize_object(field);
    }
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
