/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once

/////////////////////// StdLib includes


/////////////////////// Qt includes


/////////////////////// pappsomspp includes


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/jsclassregistrar.h"
#include "MsXpS/libXpertMassCore/globals.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{

extern int FWHM_PEAK_SPAN_FACTOR;

/*  BEGIN CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: MassPeakShaperConfig
 */

class DECLSPEC MassPeakShaperConfig: public QObject
{
  Q_OBJECT
  Q_PROPERTY(double resolution READ getResolution WRITE setResolution MEMBER
               m_resolution)
  Q_PROPERTY(double fwhm READ getFwhm WRITE setFwhm MEMBER m_fwhm)
  Q_PROPERTY(
    Enums::MassPeakWidthLogic massPeakWidthLogic READ getMassPeakWidthLogic
      WRITE setMassPeakWidthLogic MEMBER m_massPeakWidthLogic)
  Q_PROPERTY(double referencePeakMz READ getReferencePeakMz WRITE
               setReferencePeakMz MEMBER m_referencePeakMz)
  Q_PROPERTY(
    int pointCount READ getPointCount WRITE setPointCount MEMBER m_pointCount)
  Q_PROPERTY(bool withBins READ isWithBins WRITE setWithBins MEMBER m_withBins)
  Q_PROPERTY(int binSizeDivisor READ getBinSizeDivisor WRITE setBinSizeDivisor
               MEMBER m_binSizeDivisor)
  Q_PROPERTY(double binSize READ getBinSize WRITE setBinSize MEMBER m_binSize)
  Q_PROPERTY(bool isBinSizeFixed READ getBinSizeFixed WRITE setBinSizeFixed
               MEMBER m_isBinSizeFixed)
  Q_PROPERTY(double mzStep READ getMzStep WRITE setMzStep MEMBER m_mzStep)
  Q_PROPERTY(
    Enums::MassPeakShapeType massPeakShapeType READ getMassPeakShapeType WRITE
      setMassPeakShapeType MEMBER m_massPeakShapeType)

  public:
  explicit MassPeakShaperConfig(QObject *parent = nullptr);
  // Pseudo copy constructor.
  explicit MassPeakShaperConfig(const MassPeakShaperConfig &other,
                                QObject *parent = nullptr);

  virtual ~MassPeakShaperConfig();

  MassPeakShaperConfig &initialize(const MassPeakShaperConfig &other);
  MassPeakShaperConfig *clone(const MassPeakShaperConfig &other,
                              QObject *parent = nullptr) const;
  void operator=(const MassPeakShaperConfig &other) = delete;

  void setResolution(int resolution);
  double getResolution() const;
  int resolution(bool *ok);

  // For the gaussion, that is the entirety of the fwhm.
  void setFwhm(double value);
  double getFwhm() const;
  double fwhm(bool *ok);
  // For the lorentzian, that is half of the fwhm.
  double halfFwhm(bool *ok);

  void setReferencePeakMz(double mz);
  double getReferencePeakMz() const;

  void setBinSize(double bin_size);
  double binSize(bool *ok);
  double getBinSize() const;
  void setWithBins(bool with_bins);
  bool isWithBins() const;
  void setBinSizeFixed(bool is_fixed);
  bool getBinSizeFixed();

  void setBinSizeDivisor(int divisor);
  int getBinSizeDivisor() const;

  void setPointCount(int point_count);
  int getPointCount() const;

  void setMassPeakShapeType(Enums::MassPeakShapeType mass_peak_shape_type);
  Enums::MassPeakShapeType getMassPeakShapeType() const;

  void setMassPeakWidthLogic(Enums::MassPeakWidthLogic logic);
  Enums::MassPeakWidthLogic getMassPeakWidthLogic() const;

  double c(bool *ok);

  double a(bool *ok);

  double gamma(bool *ok);

  void setMzStep(double step);
  double getMzStep() const;
  double mzStep(bool *ok);

  bool resolve(libXpertMassCore::ErrorList &error_list);
  void reset();

  QString toString();

  static void registerJsConstructor(QJSEngine *engine);

  protected:
  int m_resolution = 0;
  double m_fwhm    = 0;

  Enums::MassPeakWidthLogic m_massPeakWidthLogic =
    Enums::MassPeakWidthLogic::NOT_SET;

  // The reference m/z value is the m/z value of the peak that is considered
  // to be the main peak in an isotopic cluster, for example. That is
  // typically the peak that is the most intense. This peak's intensity is
  // typically used to compute the FWHM on the basis of the resolution, for
  // example (see function fwhm().  When computing a Gaussian shape for a
  // single peak, then that reference peak's m/z value needs to be set as the
  // peak centroid for which the computation is performed.
  double m_referencePeakMz = 0;

  // Number of points to use to shape the peak
  int m_pointCount      = 0;
  bool m_withBins       = false;
  int m_binSizeDivisor  = 6;
  double m_binSize      = 0;
  bool m_isBinSizeFixed = false;
  // The delta bewteen two consecutive data points
  double m_mzStep       = 0;
  // Type (GAUSSIAN | LORENTZIAN) of the peak shape
  Enums::MassPeakShapeType m_massPeakShapeType =
    Enums::MassPeakShapeType::NOT_SET;
};

/*  END CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name:
 */

} // namespace libXpertMassCore
MSXPS_REGISTER_JS_CLASS(MsXpS::libXpertMassCore, MassPeakShaperConfig)
} // namespace MsXpS
