/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ClusterBuilder;
import org.ovirt.engine.sdk4.builders.DiskBuilder;
import org.ovirt.engine.sdk4.builders.StorageDomainBuilder;
import org.ovirt.engine.sdk4.builders.TemplateBuilder;
import org.ovirt.engine.sdk4.types.Cluster;
import org.ovirt.engine.sdk4.types.Disk;
import org.ovirt.engine.sdk4.types.Image;
import org.ovirt.engine.sdk4.types.StorageDomain;
import org.ovirt.engine.sdk4.types.Template;

public interface ImageService extends Service {
    public interface GetRequest extends Request<GetRequest, GetResponse> {
    }
    
    public interface GetResponse extends Response {
        Image image();
    }
    
    GetRequest get();
    
    /**
     * Imports an image.
     * 
     * If the `import_as_template` parameter is `true` then the image will be imported as a template, otherwise it will
     * be imported as a disk.
     * 
     * When imported as a template, the name of the template can be specified by the optional `template.name`
     * parameter. If that parameter is not specified, then the name of the template will be automatically assigned by the
     * engine as `GlanceTemplate-x` (where `x` will be seven random hexadecimal characters).
     * 
     * When imported as a disk, the name of the disk can be specified by the optional `disk.name` parameter. If
     * that parameter is not specified, then the name of the disk will be automatically assigned by the engine as
     * `GlanceDisk-x` (where `x` will be the seven hexadecimal characters of the image identifier).
     * 
     * It is recommended to always explicitly specify the template or disk name, to avoid these automatic names
     * generated by the engine.
     */
    public interface ImportRequest extends Request<ImportRequest, ImportResponse> {
        /**
         * Indicates if the import should be performed asynchronously.
         */
        ImportRequest async(Boolean async);
        ImportRequest cluster(Cluster cluster);
        /**
         * The cluster to which the image should be imported if the `import_as_template` parameter
         * is set to `true`.
         */
        ImportRequest cluster(ClusterBuilder cluster);
        ImportRequest disk(Disk disk);
        /**
         * The disk to import.
         */
        ImportRequest disk(DiskBuilder disk);
        /**
         * Specifies if a template should be created from the imported disk.
         */
        ImportRequest importAsTemplate(Boolean importAsTemplate);
        ImportRequest storageDomain(StorageDomain storageDomain);
        /**
         * The storage domain to which the disk should be imported.
         */
        ImportRequest storageDomain(StorageDomainBuilder storageDomain);
        ImportRequest template(Template template);
        /**
         * The name of the template being created if the
         * `import_as_template` parameter is set to `true`.
         */
        ImportRequest template(TemplateBuilder template);
    }
    
    /**
     * Imports an image.
     * 
     * If the `import_as_template` parameter is `true` then the image will be imported as a template, otherwise it will
     * be imported as a disk.
     * 
     * When imported as a template, the name of the template can be specified by the optional `template.name`
     * parameter. If that parameter is not specified, then the name of the template will be automatically assigned by the
     * engine as `GlanceTemplate-x` (where `x` will be seven random hexadecimal characters).
     * 
     * When imported as a disk, the name of the disk can be specified by the optional `disk.name` parameter. If
     * that parameter is not specified, then the name of the disk will be automatically assigned by the engine as
     * `GlanceDisk-x` (where `x` will be the seven hexadecimal characters of the image identifier).
     * 
     * It is recommended to always explicitly specify the template or disk name, to avoid these automatic names
     * generated by the engine.
     */
    public interface ImportResponse extends Response {
    }
    
    /**
     * Imports an image.
     * 
     * If the `import_as_template` parameter is `true` then the image will be imported as a template, otherwise it will
     * be imported as a disk.
     * 
     * When imported as a template, the name of the template can be specified by the optional `template.name`
     * parameter. If that parameter is not specified, then the name of the template will be automatically assigned by the
     * engine as `GlanceTemplate-x` (where `x` will be seven random hexadecimal characters).
     * 
     * When imported as a disk, the name of the disk can be specified by the optional `disk.name` parameter. If
     * that parameter is not specified, then the name of the disk will be automatically assigned by the engine as
     * `GlanceDisk-x` (where `x` will be the seven hexadecimal characters of the image identifier).
     * 
     * It is recommended to always explicitly specify the template or disk name, to avoid these automatic names
     * generated by the engine.
     */
    ImportRequest import_();
    
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

