/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.NetworkBuilder;
import org.ovirt.engine.sdk4.types.Network;

/**
 * A service to manage cluster networks.
 */
public interface ClusterNetworksService extends Service {
    /**
     * Assigns the network to a cluster.
     * 
     * Post a request like in the example below to assign the network to a cluster:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters/123/networks
     * ----
     * 
     * Use the following example in its body:
     * 
     * [source,xml]
     * ----
     * <network id="123" />
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest network(Network network);
        /**
         * The network object to be assigned to the cluster.
         */
        AddRequest network(NetworkBuilder network);
    }
    
    /**
     * Assigns the network to a cluster.
     * 
     * Post a request like in the example below to assign the network to a cluster:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters/123/networks
     * ----
     * 
     * Use the following example in its body:
     * 
     * [source,xml]
     * ----
     * <network id="123" />
     * ----
     */
    public interface AddResponse extends Response {
        /**
         * The network object to be assigned to the cluster.
         */
        Network network();
    }
    
    /**
     * Assigns the network to a cluster.
     * 
     * Post a request like in the example below to assign the network to a cluster:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters/123/networks
     * ----
     * 
     * Use the following example in its body:
     * 
     * [source,xml]
     * ----
     * <network id="123" />
     * ----
     */
    AddRequest add();
    
    /**
     * Lists the networks that are assigned to the cluster.
     * 
     * The order of the returned clusters isn't guaranteed.
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Sets the maximum number of networks to return. If not specified, all the networks are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of networks to return. If not specified, all the networks are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of networks to return. If not specified, all the networks are returned.
         */
        ListRequest max(BigInteger max);
    }
    
    /**
     * Lists the networks that are assigned to the cluster.
     * 
     * The order of the returned clusters isn't guaranteed.
     */
    public interface ListResponse extends Response {
        /**
         * The list of networks that are assigned to the cluster.
         */
        List<Network> networks();
    }
    
    /**
     * Lists the networks that are assigned to the cluster.
     * 
     * The order of the returned clusters isn't guaranteed.
     */
    ListRequest list();
    
    /**
     * Access the cluster network service that manages the cluster network specified by an ID.
     */
    ClusterNetworkService networkService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

