/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.StorageConnectionBuilder;
import org.ovirt.engine.sdk4.types.StorageConnection;

public interface StorageServerConnectionService extends Service {
    public interface GetRequest extends Request<GetRequest, GetResponse> {
    }
    
    public interface GetResponse extends Response {
        StorageConnection conection();
    }
    
    GetRequest get();
    
    /**
     * Removes a storage connection.
     * 
     * A storage connection can only be deleted if neither storage domain nor LUN disks reference it. The host name or
     * id is optional; providing it disconnects (unmounts) the connection from that host.
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
        /**
         * The name or identifier of the host from which the connection would be unmounted (disconnected). If not
         * provided, no host will be disconnected.
         * 
         * For example, to use the host with identifier `456` to delete the storage connection with identifier `123`
         * send a request like this:
         * 
         * [source]
         * ----
         * DELETE /ovirt-engine/api/storageconnections/123?host=456
         * ----
         */
        RemoveRequest host(String host);
    }
    
    /**
     * Removes a storage connection.
     * 
     * A storage connection can only be deleted if neither storage domain nor LUN disks reference it. The host name or
     * id is optional; providing it disconnects (unmounts) the connection from that host.
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Removes a storage connection.
     * 
     * A storage connection can only be deleted if neither storage domain nor LUN disks reference it. The host name or
     * id is optional; providing it disconnects (unmounts) the connection from that host.
     */
    RemoveRequest remove();
    
    /**
     * Updates the storage connection.
     * 
     * For example, to change the address of the storage server send a request like this:
     * 
     * [source,xml]
     * ----
     * PUT /ovirt-engine/api/storageconnections/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <storage_connection>
     *   <address>mynewnfs.example.com</address>
     *   <host>
     *     <name>myhost</name>
     *   </host>
     * </storage_connection>
     * ----
     */
    public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
        /**
         * Indicates if the update should be performed asynchronously.
         */
        UpdateRequest async(Boolean async);
        UpdateRequest connection(StorageConnection connection);
        UpdateRequest connection(StorageConnectionBuilder connection);
        /**
         * Indicates if the operation should succeed regardless to the relevant storage domain's status
         * (i.e. updating is also applicable when storage domain's status is not maintenance).
         * 
         * This parameter is optional, and the default value is `false`.
         */
        UpdateRequest force(Boolean force);
    }
    
    /**
     * Updates the storage connection.
     * 
     * For example, to change the address of the storage server send a request like this:
     * 
     * [source,xml]
     * ----
     * PUT /ovirt-engine/api/storageconnections/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <storage_connection>
     *   <address>mynewnfs.example.com</address>
     *   <host>
     *     <name>myhost</name>
     *   </host>
     * </storage_connection>
     * ----
     */
    public interface UpdateResponse extends Response {
        StorageConnection connection();
    }
    
    /**
     * Updates the storage connection.
     * 
     * For example, to change the address of the storage server send a request like this:
     * 
     * [source,xml]
     * ----
     * PUT /ovirt-engine/api/storageconnections/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <storage_connection>
     *   <address>mynewnfs.example.com</address>
     *   <host>
     *     <name>myhost</name>
     *   </host>
     * </storage_connection>
     * ----
     */
    UpdateRequest update();
    
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

